package main

import (
	"encoding/csv"
	"flag"
	"fmt"
	"io"
	"log"
	"os"
	"time"

	"code.justin.tv/availability/goracle/catalog"
	"code.justin.tv/availability/hms-esk/pkg/apihandlers"
	"code.justin.tv/availability/hms-esk/pkg/config"
	"code.justin.tv/availability/hms-esk/pkg/datasource"
)

func setupConfig() {
	if len(os.Args) == 2 {
		if err := config.ParseConfig(os.Args[1]); err != nil {
			log.Fatalf("Error reading TOML config -- %s", err)
		}
	}

}

// exapme run `go run cmd/backfill/main.go -csv Queries.csv -date 11-12-2017`
// takes a csv of the format id, query, aggregate_type which can be produced via
// select id, query, aggregate_type from catalogdb.queries where query like '%stats.counters.visage%'
func main() {
	csvPtr := flag.String("csv", "", "takes a csv of the format id, query, aggregate_type")
	datePtr := flag.String("date", "", "The date to run this for defaults to today")
	configPtr := flag.String("config", "", "The config.toml to use")
	flag.Parse()

	if *csvPtr == "" {
		fmt.Print("A csv file must be provided after -csv")
		return
	}
	date := time.Now()
	var err error
	if *datePtr == "" {
		fmt.Print("A date of the format MM-DD-YYYY must be provided after -date")
		return
	} else {
		date, err = time.Parse("01-02-2006", *datePtr)
		if err != nil {
			log.Fatalln(err)
			return
		}
	}

	fmt.Printf("date: %s\n", date)
	if *configPtr != "" {
		if err = config.ParseConfig(*configPtr); err != nil {
			log.Fatalf("Error reading TOML config -- %s", err)
			return
		}
	}
	file, err := os.Open(*csvPtr)
	if err != nil {
		log.Fatal(err)
	}
	reader := csv.NewReader(file)
	for {
		record, err := reader.Read()

		// loop termination condition 1:  EOF.
		// this is the normal loop termination condition.
		if err == io.EOF {
			fmt.Print("end of file")
			break
		}

		// loop termination condition 2: some other error.
		// Errors happen, so check for them and do something with them.
		if err != nil {
			log.Fatalln(err)
		}

		fmt.Printf("Processing %s \n", record[0])
		id, err := apihandlers.StringToUint(record[0])
		if err != nil {
			fmt.Printf("error parsing id: %s\n", record[0])
			continue
		}
		query := catalog.Query{Query: record[1], AggregateType: catalog.AggregateType(record[2])}
		query.ID = id

		err = datasource.SendQueryToEcho(&query, date)
		if err != nil {
			log.Fatalln(err)
		}

	}
}
