package api

import (
	"code.justin.tv/availability/goracle/serverutil"
	"code.justin.tv/availability/hms-esk/pkg/apihandlers"
	"code.justin.tv/availability/hms-esk/pkg/config"
	"code.justin.tv/foundation/twitchserver"
	"github.com/rs/cors"
	"github.com/sirupsen/logrus"
	goji "goji.io"
	"goji.io/pat"
)

type eskServer struct {
	*goji.Mux
}

func (s *eskServer) addHandlers() {
	s.Use(serverutil.Logger)
	mdapi := goji.SubMux()
	c := cors.New(cors.Options{
		AllowedOrigins: []string{"*"},
	})
	mdapi.Use(c.Handler)

	// Each handler will also accept the parameters start, end,
	// They will return a metricData struct
	s.Handle(pat.New("/v1/*"), mdapi)
	mdapi.HandleFunc(pat.Get("/availability/weekly/"), apihandlers.AvailablityWeeklySummaryHandler)
	mdapi.HandleFunc(pat.Get("/availability/summary/"), apihandlers.AvailabilitySummaryHandler)
	mdapi.HandleFunc(pat.Get("/availability/"), apihandlers.AvailabilityHandler)
	mdapi.HandleFunc(pat.Get("/downtime/budget/"), apihandlers.DowntimeBudgetHandler)
	mdapi.HandleFunc(pat.Get("/downtime/"), apihandlers.DowntimeHandler)
	mdapi.HandleFunc(pat.Get("/latency/summary/"), apihandlers.LatencySummaryHandler)
	mdapi.HandleFunc(pat.Get("/queries/preview/"), apihandlers.QueryDataPreviewHandler)
	mdapi.HandleFunc(pat.Get("/queries/"), apihandlers.QueryDataHandler)
	mdapi.HandleFunc(pat.Get("/rawdata/"), apihandlers.RawDataHandler)
	mdapi.HandleFunc(pat.Get("/unhealthy/"), apihandlers.UnhealthyHandler)
	mdapi.HandleFunc(pat.Get("/reset/"), apihandlers.ResetHandler)
	mdapi.HandleFunc(pat.Get("/reportitems/"), apihandlers.GetReportItemsHandler)
	mdapi.HandleFunc(pat.Post("/reportitems/"), apihandlers.PostReportItemsHandler)
}

// RunServer sets up the UI and API routes and starts the web server
func RunServer() {
	server := twitchserver.NewServer()
	dimensionKey := new(int)
	elevateKey := new(int)
	s := &eskServer{server}

	// Default to no caching for development
	s.Use(NoCache)
	s.Use(serverutil.Logger)

	s.addHandlers()
	twitchserver.AddDefaultSignalHandlers()

	conf := twitchserver.ServerConfig{
		DimensionKey: dimensionKey,
		ElevateKey:   elevateKey,
	}

	if config.Config.AppPort != "" {
		conf.Addr = config.Config.AppPort
		conf.DebugAddr = config.Config.DebugPort
	}
	err := twitchserver.ListenAndServe(s, &conf)
	if err != nil {
		logrus.Fatal(err)
	}
}
