package apihandlers

import (
	"fmt"
	"strconv"
	"time"

	"encoding/json"
	"net/http"

	"code.justin.tv/availability/hms-esk/pkg/datamanager"
	"code.justin.tv/availability/hms-esk/pkg/daydata"
	"github.com/jinzhu/now"
	"github.com/sirupsen/logrus"
)

type MetricData struct {
	// These provided only if no end time is specified
	Current    *bool    `json:"current,omitempty"`
	Today      *float64 `json:"today,omitempty"`
	WeektoDate *float64 `json:"week_to_date,omitempty"`

	Points *[]daydata.GraphDatum `json:"points,omitempty"`
}

func BulkAvailabilitySummaryHandler(w http.ResponseWriter, options map[string][]string, items map[uint]interface{}) {
	var returnJSON []byte

	currentVal, currentValok := options["current"]
	var parsedVal bool
	if currentValok {
		var err error
		// This accepts 1, t, T, TRUE, true, True, 0, f, F, FALSE, false, False
		// see https://golang.org/pkg/strconv/#ParseBool
		parsedVal, err = strconv.ParseBool(currentVal[0])
		if err != nil {
			HandleAPIErrors(w, []error{err}, 400)
			return
		}
	}
	startOfInterval := now.BeginningOfDay().Add(time.Duration(-7*24) * time.Hour)
	availabilityData := map[uint]*MetricData{}
	for id, item := range items {
		dm := datamanager.GetDataManager(item)
		current, today, weektodate := getCurrent(dm, now.BeginningOfDay())
		var availData *MetricData
		if dm.Interval(startOfInterval, 7) == nil {
			availData = nil
		} else {
			availData = &MetricData{Current: current, Today: today, WeektoDate: weektodate}
		}

		if !currentValok || *current == parsedVal {
			availabilityData[id] = availData
		}
	}

	returnJSON, err := json.Marshal(availabilityData)
	if err != nil {
		HandleAPIErrors(w, []error{err}, 500)
		return
	}

	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(200)
	w.Write(returnJSON)
}

func AvailabilitySummaryHandler(w http.ResponseWriter, r *http.Request) {
	options := r.URL.Query()
	item, err := getItem(options)
	items, err2 := getItems(options)
	if err == nil && item != nil {
		apply := func(m datamanager.MetricDataInterval) []daydata.GraphDatum {
			return m.AvailabilityPercentGraphDataPoints()
		}
		intervalCommon(w, r, true, apply)
	} else if err2 == nil && items != nil {
		BulkAvailabilitySummaryHandler(w, options, items)
	} else {
		HandleAPIErrors(w, []error{fmt.Errorf("Either a valid id or a valid list of ids must be passed")}, 500)
		return
	}

}

func AvailabilityHandler(w http.ResponseWriter, r *http.Request) {
	apply := func(m datamanager.MetricDataInterval) []daydata.GraphDatum {
		return m.AvailabilityBoolGraphDataDetailed()
	}
	intervalCommon(w, r, true, apply)
}

func intervalCommon(w http.ResponseWriter, r *http.Request, avail bool, apply func(datamanager.MetricDataInterval) []daydata.GraphDatum) {
	var returnJSON []byte
	options := r.URL.Query()
	endDate, startDate, errs := parseCommon(options)
	if len(errs) > 0 {
		HandleAPIErrors(w, errs, 400)
		return
	}
	item, err := getItem(options)
	if err != nil {
		logrus.Info("Unable to find the item requested.")
		HandleAPIErrors(w, []error{err}, 404)
		return
	}
	dm := datamanager.GetDataManager(item)
	availabilityData := &MetricData{}
	if avail {
		current, today, weektodate := getCurrent(dm, endDate)
		availabilityData = &MetricData{Current: current, Today: today, WeektoDate: weektodate}
	}

	interval, err := getInterval(dm, startDate, endDate)
	if err != nil {
		HandleAPIErrors(w, []error{err}, 400)
		return
	}
	if interval == nil {
		// No data available, but not an error. Return null
		returnJSON, err = json.Marshal(nil)
		if err != nil {
			HandleAPIErrors(w, []error{err}, 500)
			return
		}
	} else {
		if interval.Days() != nil {
			data := apply(interval)
			availabilityData.Points = &data
		}

		returnJSON, err = json.Marshal(availabilityData)
		if err != nil {
			HandleAPIErrors(w, []error{err}, 500)
			return
		}
	}
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(200)
	w.Write(returnJSON)
}

func getCurrent(dm datamanager.DataManager, endDate time.Time) (*bool, *float64, *float64) {
	if endDate.After(now.BeginningOfDay()) || endDate.Equal(now.BeginningOfDay()) {
		var current bool
		var today, weekToDate float64
		if dm.Today() != nil {
			current = dm.Today().RecentAvailability()
			today = dm.Today().Availability()
		}
		week, _ := dm.Weeks(1)
		if len(week) > 0 {
			weekToDate = week[0].Availability()
		}
		return &current, &today, &weekToDate
	}
	return nil, nil, nil
}

// AvailablityWeeklySummaryHandler handles request for weekly data
func AvailablityWeeklySummaryHandler(w http.ResponseWriter, r *http.Request) {
	var returnJSON []byte

	options := r.URL.Query()
	_, _, errs := parseCommon(options)
	if len(errs) > 0 {
		HandleAPIErrors(w, errs, 400)
		return
	}
	if len(errs) > 0 {
		HandleAPIErrors(w, errs, 400)
		return
	}
	item, err := getItem(options)
	if err != nil {
		logrus.Info("Unable to find the item requested.")
		HandleAPIErrors(w, []error{err}, 404)
		return
	}
	dm := datamanager.GetDataManager(item)
	weeks, err := dm.Weeks(5)

	weeksPoints := []daydata.FloatDatum{}
	for _, w := range weeks {
		weeksPoints = append(weeksPoints, daydata.FloatDatum{
			Time:  w.StartDate(),
			Value: w.Availability(),
		})

	}

	returnJSON, err = json.Marshal(weeksPoints)
	if err != nil {
		HandleAPIErrors(w, []error{err}, 500)
		return
	}
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(200)
	w.Write(returnJSON)
}
