package apihandlers

import (
	"encoding/json"
	"net/http"

	"github.com/sirupsen/logrus"

	"code.justin.tv/availability/hms-esk/pkg/datamanager"
	"code.justin.tv/availability/hms-esk/pkg/daydata"
)

type LatencySummaryData struct {
	P50Weeks *[]daydata.FloatDatum `json:"p50_weeks,omitempty"`
	P90Weeks *[]daydata.FloatDatum `json:"p90_weeks,omitempty"`
	P99Weeks *[]daydata.FloatDatum `json:"p99_weeks,omitempty"`
}

// LatencySummaryHandler handles
func LatencySummaryHandler(w http.ResponseWriter, r *http.Request) {
	var returnJSON []byte

	options := r.URL.Query()
	_, _, errs := parseCommon(options)
	if len(errs) > 0 {
		HandleAPIErrors(w, errs, 400)
		return
	}
	if len(errs) > 0 {
		HandleAPIErrors(w, errs, 400)
		return
	}
	item, err := getItem(options)
	if err != nil {
		logrus.Info("Unable to find the item requested.")
		HandleAPIErrors(w, []error{err}, 404)
		return
	}
	dm := datamanager.GetDataManager(item)
	weeks, err := dm.Weeks(5)

	p50weeks := []daydata.FloatDatum{}
	p90weeks := []daydata.FloatDatum{}
	p99weeks := []daydata.FloatDatum{}
	for _, w := range weeks {
		p50weeks = append(p50weeks, daydata.FloatDatum{
			Time:  w.StartDate(),
			Value: w.Latency("percentile_50"),
		})
		p90weeks = append(p90weeks, daydata.FloatDatum{
			Time:  w.StartDate(),
			Value: w.Latency("percentile_90"),
		})
		p99weeks = append(p99weeks, daydata.FloatDatum{
			Time:  w.StartDate(),
			Value: w.Latency("percentile_99"),
		})
	}

	latencySummary := &LatencySummaryData{
		P50Weeks: &p50weeks,
		P90Weeks: &p90weeks,
		P99Weeks: &p99weeks,
	}
	returnJSON, err = json.Marshal(latencySummary)
	if err != nil {
		HandleAPIErrors(w, []error{err}, 500)
		return
	}
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(200)
	w.Write(returnJSON)
}
