package apihandlers

import (
	"encoding/json"
	"fmt"
	"html/template"
	"net/http"
	"time"

	"code.justin.tv/availability/goracle/catalog"
	"code.justin.tv/availability/hms-esk/pkg/datasource"
	"code.justin.tv/availability/hms-esk/pkg/daydata"
	"code.justin.tv/availability/hms-esk/pkg/graphitedata"
	"github.com/sirupsen/logrus"
)

func QueryDataPreviewHandler(w http.ResponseWriter, r *http.Request) {
	queryParams := r.URL.Query()
	query := queryParams["query"][0]
	logrus.Info(query)
	queryPreviewCommon(w, r, query)
}

func QueryDataHandler(w http.ResponseWriter, r *http.Request) {
	//eventualy we will want this to check dynamo but for now we just do the same as preview
	options := r.URL.Query()
	query, err := getQuery(options)
	if err != nil {
		logrus.Warnf("Unable to find the item requested.")
		HandleAPIErrors(w, []error{err}, 404)
		return
	}
	queryPreviewCommonQuery(w, r, query)
}

func queryPreviewCommon(w http.ResponseWriter, r *http.Request, query string) {
	options := r.URL.Query()
	endDate, startDate, errs := parseCommon(options)
	if len(errs) > 0 {
		HandleAPIErrors(w, errs, 400)
		return
	}
	returnData, err := graphitedata.FetchGraphiteQueryData(query, startDate, endDate)
	if err != nil {
		logrus.Info("Failed to fetch query data for ", query, err)
		HandleAPIErrors(w, []error{err}, 400)
		return
	}

	rawData := daydata.NewRawData("Preview", query, "", "")

	// Iterate through return data and generate graph data
	for _, point := range returnData.Data {
		if point.Value != nil {
			t := time.Unix(int64(*point.Timestamp), 0)
			v := *point.Value
			rawData.Datapoints = append(rawData.Datapoints, daydata.FloatDatum{
				Time:  t,
				Value: v,
			})
		}
	}
	data := rawData.GraphDataPoints()
	returnJSON, err := json.Marshal(data)
	if err != nil {
		HandleAPIErrors(w, []error{err}, 500)
		return
	}
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(200)
	w.Write([]byte(template.JS(string(returnJSON))))
}

func queryPreviewCommonQuery(w http.ResponseWriter, r *http.Request, query *catalog.Query) {
	options := r.URL.Query()
	_, startDate, errs := parseCommon(options)
	if len(errs) > 0 {
		HandleAPIErrors(w, errs, 400)
		return
	}
	returnData, err := datasource.FetchDataForQuery(query, startDate)
	if err != nil {
		logrus.Info("Failed to fetch query data for ", query, err)
		HandleAPIErrors(w, []error{err}, 400)
		return
	}

	rawData := daydata.NewRawData("Preview", query.Query, "", "")

	// Iterate through return data and generate graph data
	for _, point := range returnData.Data {
		if point.Value != nil {
			t := time.Unix(int64(*point.Timestamp), 0)
			v := *point.Value
			rawData.Datapoints = append(rawData.Datapoints, daydata.FloatDatum{
				Time:  t,
				Value: v,
			})
		}
	}
	data := rawData.GraphDataPoints()
	returnJSON, err := json.Marshal(data)
	if err != nil {
		HandleAPIErrors(w, []error{err}, 500)
		return
	}
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(200)
	w.Write([]byte(template.JS(string(returnJSON))))
}

func getQuery(options map[string][]string) (*catalog.Query, error) {
	if len(options["query"]) == 1 {
		id, _ := StringToUint(options["query"][0])
		item, err := catalog.GetCatalog().GetQueryByID(id)
		return item, err
	}
	return nil, fmt.Errorf("A query id must be passed.")
}
