package apihandlers

import (
	"encoding/json"
	"fmt"
	"net/http"

	"github.com/sirupsen/logrus"

	"code.justin.tv/availability/goracle/catalog"
	"code.justin.tv/availability/hms-esk/pkg/datamanager"
	"code.justin.tv/availability/hms-esk/pkg/daydata"
)

type rawData struct {
	ID          *string               `json:"id"`
	Description *string               `json:"description"`
	Label       *string               `json:"label"`
	Unit        *string               `json:"unit"`
	Points      *[]daydata.GraphDatum `json:"points"`
}

func RawDataHandler(w http.ResponseWriter, r *http.Request) {
	options := r.URL.Query()
	endDate, startDate, errs := parseCommon(options)
	if len(errs) > 0 {
		HandleAPIErrors(w, errs, 400)
		return
	}
	met, err := getMetric(options)
	if err != nil {
		logrus.Warnf("Unable to find the item requested.")
		HandleAPIErrors(w, []error{err}, 404)
		return
	}
	dm := datamanager.GetDataManager(met)
	apiData := []*rawData{}
	days := daysDiff(endDate, startDate) + 1
	interval := dm.Interval(startDate, days)
	if interval != nil && interval.Days() != nil {
		for _, data := range interval.RawData() {
			points := data.GraphDataPoints()
			tempData := rawData{
				ID:          &data.ID,
				Description: &data.Description,
				Label:       &data.Label,
				Unit:        &data.Unit,
				Points:      &points,
			}
			apiData = append(apiData, &tempData)
		}
	}
	returnJSON, err := json.Marshal(apiData)
	if err != nil {
		HandleAPIErrors(w, []error{err}, 500)
		return
	}
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(200)
	w.Write(returnJSON)
}
func getMetric(options map[string][]string) (*catalog.Metric, error) {
	if len(options["metric"]) == 1 {
		id, _ := StringToUint(options["metric"][0])
		item, err := catalog.GetCatalog().GetMetricByID(id)
		return item, err
	}
	return nil, fmt.Errorf("A metric id must be passed.")
}
