package apihandlers

import (
	"encoding/json"
	"errors"
	"fmt"
	"io/ioutil"
	"net/http"

	"code.justin.tv/availability/goracle/serverutil"
	"code.justin.tv/availability/hms-esk/pkg/reportdata"
	"github.com/sirupsen/logrus"
)

func GetReportItemsHandler(w http.ResponseWriter, r *http.Request) {
	options := r.URL.Query()
	if len(options["reportID"]) == 0 {
		HandleAPIErrors(w, []error{fmt.Errorf("A reportID must be specified")}, 400)
	}
	reportID := options["reportID"][0]
	reportItems, err := reportdata.GetReportItems(reportID)
	if err != nil {
		HandleAPIErrors(w, []error{err}, 500)
	}
	returnJSON, err := json.Marshal(reportItems)
	if err != nil {
		HandleAPIErrors(w, []error{err}, 500)
		return
	}
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(200)
	w.Write(returnJSON)
}

func PostReportItemsHandler(w http.ResponseWriter, r *http.Request) {
	var ri reportdata.ReportItem
	body, err := ioutil.ReadAll(r.Body)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{errors.New("internal server error reading HTTP POST body")}, 500)
		return
	}
	defer r.Body.Close()
	// unmarshal into an API component object
	if err = json.Unmarshal(body, &ri); err != nil {
		logrus.Warnf("Unable to unmarshal POST data: %s", err.Error())
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}
	if err = ri.Save(); err != nil {
		logrus.Warnf("Unable to save ReportItem: %s", err.Error())
		serverutil.HandleAPIErrors(w, []error{err}, 400)
		return
	}

	returnJSON, err := json.Marshal(ri)
	if err != nil {
		serverutil.HandleAPIErrors(w, []error{err}, 500)
		return
	}
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(http.StatusCreated)
	w.Write(returnJSON)
}
