package apihandlers

import (
	"encoding/json"
	"fmt"
	"net/http"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/availability/goracle/catalog"
	"code.justin.tv/availability/hms-esk/pkg/datamanager"
	"github.com/jinzhu/now"
	"github.com/sirupsen/logrus"
)

func parseCommon(options map[string][]string) (time.Time, time.Time, []error) {
	var endDate, startDate time.Time

	// Get local timezone for time parsing
	loc := time.Now().Location()

	errs := []error{}
	if val, ok := options["end"]; ok {
		var err error
		endDate, err = time.ParseInLocation("2006-01-02", val[0], loc)
		if err != nil {
			logrus.Warnf("Unable to parse end date: %s", val[0])
			errs = append(errs, err)
		}
	} else {
		endDate = now.BeginningOfDay()
	}
	if val, ok := options["start"]; ok {
		var err error
		startDate, err = time.ParseInLocation("2006-01-02", val[0], loc)
		if err != nil {
			logrus.Warnf("Unable to parse start date: %s", val[0])
			errs = append(errs, err)
		}
	} else {
		startDate = endDate.Add(time.Duration(-35*24) * time.Hour)
	}
	return endDate, startDate, errs
}

func getInterval(dm datamanager.DataManager, startDate, endDate time.Time) (datamanager.MetricDataInterval, error) {
	days := daysDiff(endDate, startDate) + 1
	logrus.Warn("days ", days)
	// The plus one is so that we the end date is included
	interval := dm.Interval(startDate, days)
	if interval == nil {
		// err := fmt.Errorf("the requested data is not loaded")
		return nil, nil
	}
	return interval, nil
}

func getItem(options map[string][]string) (interface{}, error) {
	switch {
	case len(options["feature"]) == 1:
		id, _ := StringToUint(options["feature"][0])
		item, err := catalog.GetCatalog().GetFeatureByID(id)
		return item, err
	case len(options["service"]) == 1:
		id, _ := StringToUint(options["service"][0])
		item, err := catalog.GetCatalog().GetServiceByID(id)
		return item, err
	case len(options["component"]) == 1:
		id, _ := StringToUint(options["component"][0])
		item, err := catalog.GetCatalog().GetComponentByID(id)
		return item, err
	case len(options["metric"]) == 1:
		id, _ := StringToUint(options["metric"][0])
		item, err := catalog.GetCatalog().GetMetricByID(id)
		return item, err

	}
	return nil, fmt.Errorf("An id must be passed.")
}

func getItems(options map[string][]string) (map[uint]interface{}, error) {
	switch {
	case len(options["features"]) == 1:
		var features []*catalog.Feature
		var err error
		if options["features"][0] == "all" {
			features, err = catalog.GetCatalog().GetFeatures()
			if err != nil {
				return nil, err
			}
		} else {
			return nil, fmt.Errorf("features must be all")
		}
		var interfaceSlice map[uint]interface{} = make(map[uint]interface{}, len(features))
		for _, d := range features {
			interfaceSlice[d.ID] = d
		}
		return interfaceSlice, nil
	case len(options["services"]) == 1:
		var services []*catalog.Service
		var err error
		if options["services"][0] == "all" {
			services, err = catalog.GetCatalog().GetServices(nil)
			if err != nil {
				return nil, err
			}
		} else {
			ids, err := StringToUints(options["services"][0])
			if err != nil {
				return nil, fmt.Errorf("services must be a comma separated list of ids or all")
			}
			services, err = catalog.GetCatalog().GetServicesByIDs(ids)
			if err != nil {
				return nil, err
			}
		}
		var interfaceSlice map[uint]interface{} = make(map[uint]interface{}, len(services))
		for _, d := range services {
			interfaceSlice[d.ID] = d
		}
		return interfaceSlice, nil
	case len(options["components"]) == 1:
		var components []*catalog.Component
		var err error
		if options["components"][0] == "all" {
			components, err = catalog.GetCatalog().GetComponents(nil)
			if err != nil {
				return nil, err
			}
		} else {
			ids, err := StringToUints(options["components"][0])
			if err != nil {
				return nil, fmt.Errorf("components must be a comma separated list of ids or all")
			}
			components, err = catalog.GetCatalog().GetComponentsByIDs(ids)
			if err != nil {
				return nil, err
			}
		}
		var interfaceSlice map[uint]interface{} = make(map[uint]interface{}, len(components))
		for _, d := range components {
			interfaceSlice[d.ID] = d
		}
		return interfaceSlice, nil
	case len(options["metrics"]) == 1:
		metrics := []*catalog.Metric{}
		if options["metrics"][0] == "all" {
			metrics = catalog.GetCatalog().GetMetrics(nil)
		} else {
			ids, err := StringToUints(options["metrics"][0])
			if err != nil {
				return nil, fmt.Errorf("metrics must be a comma separated list of ids or all")
			}
			metrics, err = catalog.GetCatalog().GetMetricsByIDs(ids)
			if err != nil {
				return nil, err
			}
		}
		var interfaceSlice map[uint]interface{} = make(map[uint]interface{}, len(metrics))
		for _, d := range metrics {
			interfaceSlice[d.ID] = d
		}
		return interfaceSlice, nil

	}
	return nil, fmt.Errorf("ids must be passed.")
}

func StringToUints(in string) ([]uint, error) {
	ids := []uint{}
	idStrings := strings.Split(in, ",")
	for _, idString := range idStrings {
		out, err := strconv.ParseUint(idString, 10, 64)
		if err != nil {
			return nil, err
		}
		ids = append(ids, uint(out))
	}

	return ids, nil
}

func StringToUint(in string) (uint, error) {
	out, err := strconv.ParseUint(in, 10, 64)
	if err != nil {
		return 0, err
	}
	return uint(out), nil
}

// APIErrorMsg is a convenience struct
// used for solely for marshaling to JSON
// to create a consistently formatted response
// in the event of API errors
type APIErrorMsg struct {
	Errors []string `json:"errors"`
}

// HandleAPIErrors will take a response writer, a list of errors,
// and a desired HTTP response code and will craft a response to the client
// containing well formatted error descriptions.
// Use this method for consistent error message formats across the API
func HandleAPIErrors(w http.ResponseWriter, errs []error, statusCode int) {
	msgs := make([]string, 0)
	for _, err := range errs {
		msgs = append(msgs, err.Error())
	}
	apiErr := APIErrorMsg{Errors: msgs}
	// Try to marshal the error message for response
	errMsg, err := json.Marshal(apiErr)
	if err != nil {
		// hopefully this never happens, but it could!
		statusCode = 500
		errMsg = []byte("{\"errors\": [\"internal server error\"]")
	}
	// We could probably check errors here, but we are already in error handling
	// land so why even bother, just try to get the client an error message
	// in a best effort fashion.
	w.Header().Set("Content-Type", "application/json; charset=UTF-8")
	w.WriteHeader(statusCode)
	w.Write(errMsg)
}

// This truncates the number of days
// Same days will return 0
func daysDiff(a, b time.Time) int {
	logrus.Info((now.New(a).BeginningOfDay().Sub(now.New(b).BeginningOfDay()).Hours()) / 24)
	days := int((now.New(a).BeginningOfDay().Sub(now.New(b).BeginningOfDay()).Hours()) / 24)
	return days
}
