package apihandlers

import (
	"fmt"
	"testing"
	"time"

	"code.justin.tv/availability/hms-esk/pkg/datamanager"
	"code.justin.tv/availability/hms-esk/pkg/datasource"

	"github.com/jinzhu/now"
	"github.com/sirupsen/logrus"
	"github.com/stretchr/testify/assert"
)

// Tests for Parse Common
func TestParseCommonNoOptions(t *testing.T) {
	endDate, startDate, errs := parseCommon(map[string][]string{})
	assert.Equal(t, now.BeginningOfDay(), endDate, "End Date should be equal to the beginning of today")
	assert.Equal(t, endDate.Add(time.Hour*-35*24), startDate, "Start Date should be 35 days before End Date")
	assert.Len(t, errs, 0, "There should be no errors")
}

func TestParseCommonOnlyEnd(t *testing.T) {
	loc := time.Now().Location()
	endString := "2017-05-01"
	endExpected, _ := time.ParseInLocation("2006-01-02", endString, loc)
	endDate, startDate, errs := parseCommon(map[string][]string{"end": []string{endString}})
	assert.Equal(t, endExpected, endDate, "End Date should be equal to the beginning of today")
	assert.Equal(t, endExpected.Add(time.Hour*-35*24), startDate, "Start Date should be 35 days before End Date")
	assert.Len(t, errs, 0, "There should be no errors")
}

func TestParseCommonOnlyStart(t *testing.T) {
	loc := time.Now().Location()
	startString := "2017-05-01"
	startExpected, _ := time.ParseInLocation("2006-01-02", startString, loc)
	endDate, startDate, errs := parseCommon(map[string][]string{"start": []string{startString}})
	assert.Equal(t, now.BeginningOfDay(), endDate, "End Date should be equal to the beginning of today")
	assert.Equal(t, startExpected, startDate, "Start Date should be 35 days before End Date")
	assert.Len(t, errs, 0, "There should be no errors")
}

func TestParseCommonEndAndStart(t *testing.T) {
	loc := time.Now().Location()
	startString := "2017-05-01"
	startExpected, _ := time.ParseInLocation("2006-01-02", startString, loc)
	endString := "2017-05-02"
	endExpected, _ := time.ParseInLocation("2006-01-02", endString, loc)
	endDate, startDate, errs := parseCommon(map[string][]string{"end": []string{endString}, "start": []string{startString}})
	assert.Equal(t, endExpected, endDate, "End Date should be equal to the beginning of today")
	assert.Equal(t, startExpected, startDate, "Start Date should be 35 days before End Date")
	assert.Len(t, errs, 0, "There should be no errors")
}

func TestParseCommonErrors(t *testing.T) {
	startString := "2017-05-02a"
	endString := "2017-05-01a"
	_, _, errs := parseCommon(map[string][]string{"end": []string{endString}, "start": []string{startString}})
	assert.Len(t, errs, 2, "There should be 2 errors")
}

// Tests for daysDiff

func TestDaysDiffSame(t *testing.T) {
	days := daysDiff(now.BeginningOfDay(), now.BeginningOfDay())
	assert.Zero(t, days, "the same day should return 0 days")
}

func TestDaysDiff35Days(t *testing.T) {
	days := daysDiff(now.BeginningOfDay(), now.BeginningOfDay().Add(time.Hour*-35*24))
	assert.Equal(t, 35, days, "the same day should return 0 days")
}

// Tests for getInterval

func TestGetInterval(t *testing.T) {
	startDate := now.BeginningOfDay().Add(time.Hour * -7 * 24)
	endDate := now.BeginningOfDay()
	logrus.Info(startDate, endDate)
	dm := datamanager.NewDataManagerTest(1, "string", datasource.NewMockDataSource(1, "string"), 35)
	interval, _ := getInterval(dm, startDate, endDate)

	assert.NotNil(t, interval, "there should be dates in this interval")
	assert.Equal(t, fmt.Sprintf("%s - %s", startDate.Format("2006-01-02"), endDate.Format("2006-01-02")), interval.DateRangeStr(), "The interval has the wrong start or end ")
}

func TestGetIntervalOldDates(t *testing.T) {
	startDate := now.BeginningOfDay().Add(time.Hour * -40 * 24)
	endDate := now.BeginningOfDay().Add(time.Hour * -39 * 24)
	dm := datamanager.NewDataManagerTest(1, "string", datasource.NewMockDataSource(1, "string"), 35)
	interval, _ := getInterval(dm, startDate, endDate)

	assert.Nil(t, interval, "There should be no data")
}

// Tests for StringToUint

func TestStringToUint(t *testing.T) {
	id, err := StringToUint("12")
	assert.NoError(t, err, "there should be no error")
	assert.Equal(t, uint(12), id, "the id should be 12")

	id, err = StringToUint("a")
	assert.Error(t, err, "there should be an error")
	assert.Equal(t, uint(0), id, "the id should be 0")
}

// Tests for getItem

func TestGetItemNoID(t *testing.T) {
	item, err := getItem(map[string][]string{"end": []string{"endString"}, "start": []string{"startString"}})
	assert.Error(t, err, "there should be an error when no id is passed")
	assert.Nil(t, item, "The item should be nil when no id is passed")
}
