package datamanager

import (
	"time"

	"code.justin.tv/availability/hms-esk/pkg/daydata"
)

// CombineMetricData takes MetricData from multiple sources and
// coalesces it together into a single data source
func CombineMetricData(url string, managers []DataManager) DataManager {
	// Find the earliest day, iterate through now
	firstDate := time.Now()
	for _, m := range managers {
		f := m.FirstDataDate()
		if f.Before(firstDate) {
			firstDate = f
		}
	}

	newDayData := []daydata.DayData{}

	// Iterate through all past days until now
	lastDay := time.Now()
	for curDay := firstDate; curDay.Before(lastDay); curDay = curDay.Add(time.Duration(24) * time.Hour) {
		// Get all data that we have for that day
		dayData := []daydata.DayData{}
		for _, m := range managers {
			dd := m.Day(curDay)
			if dd != nil {
				dayData = append(dayData, dd)
			}
		}

		// Now combine all of that day's data
		combinedDayData := daydata.CombineMetricDataDay(url, dayData)
		if combinedDayData != nil {
			newDayData = append(newDayData, combinedDayData)
		}
	}

	return newDataManagerFromDayData(url, newDayData)
}

func AvailabilityPercentage(days []daydata.DayData) float64 {
	// Calculate overall availability for a series of days
	minutesDown := 0
	minutesTotal := 0
	for _, day := range days {
		// Weeks can have nil days if they don't have data for that day
		if day != nil {
			minutesDown += day.MinutesDown()
			minutesTotal += day.MinutesTotal()
		}
	}

	if minutesTotal == 0 {
		return -1.0
	}

	return 100.0 * (float64(minutesTotal-minutesDown) / float64(minutesTotal))
}
