package datamanager

import (
	"math"
	"sort"
	"sync"
	"time"

	"github.com/sirupsen/logrus"
	"github.com/jinzhu/now"

	"code.justin.tv/availability/goracle/catalog"
	"code.justin.tv/availability/goracle/dateutil"
	"code.justin.tv/availability/hms-esk/pkg/datasource"
	"code.justin.tv/availability/hms-esk/pkg/daydata"
	"code.justin.tv/availability/hms-esk/pkg/config"
)

// DataManager stores "all" availability data for a time interval
type DataManager interface {
	MetricID() uint

	// This is used by email reports but should go away when datamanger is split off
	URL() string

	// Data access functions
	Source() datasource.DataSource
	AllDays() MetricDataInterval
	Day(date time.Time) daydata.DayData
	Week(date time.Time) MetricDataInterval
	// Month(date time.Time) MetricDataInterval
	Today() daydata.DayData
	Recent(numDays int) MetricDataInterval
	Interval(date time.Time, numDays int) MetricDataInterval

	// Data update APIs
	Update() // Updates the most recent data
	UpdateDataDay(date time.Time)
	UpdateDaysAgo(numdays int)
	UpdateInvaild()
	UpdateDataDays(days int)

	// Calculations based on data
	FirstDataDate() time.Time
	Weeks(numWeeks int) ([]MetricDataInterval, error)

	// returns creation time
	CreatedAt() time.Time
}

var datamanagers map[uint]*DataManager
var datamanagers_lock sync.RWMutex

func InitDataManagers() {
	datamanagers = make(map[uint]*DataManager)
}

// DataManagerImpl implements the DataManager interface
type DataManagerImpl struct {
	metricID   uint
	url        string
	dataSource datasource.DataSource
	days       []daydata.DayData
	test       bool
	createdAt  time.Time
}

// NewDataManager instantiates a MetricData with numDays of data cached
func NewDataManager(metricID uint, url string, dataSource datasource.DataSource, numDays int) DataManager {
	mdi := &DataManagerImpl{
		metricID:   metricID,
		dataSource: dataSource,
		days:       []daydata.DayData{},
		url:        url,
		test:       false,
		createdAt:  time.Now(),
	}

	mdi.updateDaysAgo(numDays)

	return mdi
}

// NewDataManager instantiates a MetricData with numDays of data cached
func NewDataManagerTest(metricID uint, url string, dataSource datasource.DataSource, numDays int) DataManager {
	mdi := &DataManagerImpl{
		metricID:   metricID,
		dataSource: dataSource,
		days:       []daydata.DayData{},
		url:        url,
		test:       true,
		createdAt:  time.Now(),
	}

	mdi.updateDaysAgo(numDays)

	return mdi
}
func newDataManagerFromDayData(url string, days []daydata.DayData) DataManager {
	return &DataManagerImpl{
		metricID:  0,
		url:       url,
		days:      days,
		createdAt: time.Now(),
	}
}

func (m *DataManagerImpl) MetricID() uint {
	return m.metricID
}

func (m *DataManagerImpl) URL() string {
	return m.url
}

func (m *DataManagerImpl) Source() datasource.DataSource {
	return m.dataSource
}

func (m *DataManagerImpl) UpdateDataDay(date time.Time) {
	// Get a new MetricDataDay for the date
	mdd := m.dataSource.DayData(date)
	if mdd == nil {
		return
	}

	// FIXME: Pretty sure that this iteration is totally unsafe
	// Find an existing dataday
	for i, d := range m.days {
		if d != nil && d.Date().Equal(date) {
			m.days[i] = mdd
			return
		}
	}
	m.days = append(m.days, mdd)
}

func (m *DataManagerImpl) UpdateDataDays(days int) {
	// Get a new MetricDataDay for the date
	mdsd := m.dataSource.DaysData(days)
	if mdsd == nil {
		logrus.Warnln("updating days nil ", m.metricID)
		return
	}
	// FIXME: Pretty sure that this iteration is totally unsafe
	// Find an existing dataday
	for i, d := range m.days {
		if d != nil {
			mdd, exists := mdsd[d.Date()]
			if exists {
				m.days[i] = mdd
				delete(mdsd, d.Date())
			}
		}

	}
	for _, day := range mdsd {
		m.days = append(m.days, day)
	}

}

func (m *DataManagerImpl) Update() {
	// TODO: Doing 2 days for rollover paranoia, can usually do 1 except around
	// midnight
	m.updateDaysAgo(2)
}

func (mdi *DataManagerImpl) updateDaysAgo(numDays int) {

	AddMultipleUpdateJob(numDays, mdi, !mdi.test)

}
func (mdi *DataManagerImpl) UpdateDaysAgo(numDays int) {
	mdi.updateDaysAgo(numDays)
}

// Day returns the metric data for the day, or nil otherwise.
// Should return an error if there was a system error that caused failure
// to retrieve data
func (m *DataManagerImpl) Day(date time.Time) daydata.DayData {
	for _, day := range m.days {
		if (day != nil) && day.Date().Format("2006-01-02") == date.Format("2006-01-02") {
			return day
		}
	}

	return nil
}

func (m *DataManagerImpl) Today() daydata.DayData {
	today := time.Now()
	for _, day := range m.days {
		if (day != nil) && day.Date().Format("2006-01-02") == today.Format("2006-01-02") {
			return day
		}
	}

	return nil
}

func (m *DataManagerImpl) Week(date time.Time) MetricDataInterval {
	return NewMetricDataInterval(date, 7, m)
}

// Recent returns the most recent <n> days of metric data
func (m *DataManagerImpl) Recent(numDays int) MetricDataInterval {
	offset := numDays - 1 // Because 1 includes today
	start := now.BeginningOfDay().Add(time.Duration(-offset*24) * time.Hour)
	return NewMetricDataInterval(start, numDays, m)
}

func (m *DataManagerImpl) Interval(date time.Time, numDays int) MetricDataInterval {
	return NewMetricDataInterval(date, numDays, m)
}

func (m *DataManagerImpl) Weeks(numWeeks int) ([]MetricDataInterval, error) {

	// Start with current week, go one week back at a time getting start of week
	// Get start of the current week

	// Pedantic note: this calculation can be off if we have leap seconds
	weekStart := dateutil.BeginningOfISOWeek(time.Now())

	weeks := []MetricDataInterval{}
	for i := 0; i < numWeeks; i++ {
		week := m.Week(weekStart)
		if week != nil {
			weeks = append(weeks, week)
		}
		weekStart = weekStart.Add(time.Duration(-24*7) * time.Hour)
	}
	return weeks, nil
}

// ByDate allows you to sort DayData by the date
type ByDate []daydata.DayData

func (b ByDate) Len() int {
	return len(b)
}

func (b ByDate) Swap(i, j int) {
	b[i], b[j] = b[j], b[i]
}

func (b ByDate) Less(i, j int) bool {
	return b[i].Date().Before(b[j].Date())
}

func (m *DataManagerImpl) AllDays() MetricDataInterval {
	// FIXME: Assumes that we will have all data up until today, also does not deal with
	// weirdnesses like leap seconds
	fd := m.FirstDataDate()
	now := now.BeginningOfDay()
	dt := now.Sub(fd)

	numDays := int(math.Floor(dt.Hours()/24) + 1)
	sort.Sort(ByDate(m.days))
	return NewMetricDataInterval(fd, numDays, m)
}

func (m *DataManagerImpl) FirstDataDate() time.Time {
	first := now.BeginningOfDay()
	for _, day := range m.days {
		if (day != nil) && day.Date().Before(first) {
			first = day.Date()
		}
	}
	return first
}
func RestartDataManagerMetric(m *catalog.Metric) {
	datamanagers_lock.Lock()
	delete(datamanagers, m.ID)
	datamanagers_lock.Unlock()
	GetDataManagerMetric(m)
}
func (m *DataManagerImpl) UpdateInvaild() {
	for _, day := range m.days {
		if day != nil && !day.IsValidDay() {
			AddUpdateJob(day.Date(), m, !m.test)
		}
	}
}

// returns creation time
func (m *DataManagerImpl) CreatedAt() time.Time {
	return m.createdAt
}

// GetDataManagerMetric returns the MetricData associated with the metric.
func GetDataManagerMetric(m *catalog.Metric) DataManager {
	if m == nil {
		logrus.Println("nil metric can not be used to produce an datamanager")
		return nil
	}
	var dataSource datasource.DataSource
	datamanagers_lock.RLock()
	dm, exists := datamanagers[m.ID]
	datamanagers_lock.RUnlock()
	stale := false
	if exists {
		if m.UpdatedAt.After((*dm).CreatedAt()) {
			stale = true
		}
		for _, q := range m.Queries {
			if q.UpdatedAt.After((*dm).CreatedAt()) {
				stale = true
			}

		}
	}
	if !exists || stale {
		if stale {
			logrus.Warnf("datamanger invalidating metric %d with dm.CreatedAt %v and metric.UpdatedAt %v", m.ID, (*dm).CreatedAt(), m.UpdatedAt)
		}
		save := true
		switch m.CalculationType {
		case catalog.Mock:
			dataSource = datasource.NewMockDataSource(m.ID, m.URL())
		case catalog.ErrorRate:
			dataSource = datasource.NewGraphiteErrorDataSource(m.ID, m.URL(), m.Queries, m.Threshold)
		case catalog.ThresholdAbove:
			dataSource = datasource.NewGraphiteThresholdDataSource(m.ID, m.URL(), m.Queries, m.Threshold, datasource.ABOVE, datasource.Above)
		case catalog.ThresholdBelow:
			dataSource = datasource.NewGraphiteThresholdDataSource(m.ID, m.URL(), m.Queries, m.Threshold, datasource.BELOW, datasource.Below)
		case catalog.Latency:
			dataSource = datasource.NewGraphiteLatencyDataSource(m.ID, m.URL(), m.LatencyQuery, m.Threshold, m.Queries)
		case "":
			logrus.Warnf("Nil metric calculation type: %s ", m.Name)
			dataSource = datasource.NewNilDataSource(m.ID, m.URL())
			save = false
		default:
			logrus.Warnf("Unknown metric calculation type: %s %s", m.CalculationType, m.Name)
			dataSource = datasource.NewNilDataSource(m.ID, m.URL())
			save = false
		}

		dmtemp := NewDataManager(m.ID, m.URL(), dataSource, DaysHistory())

		dm = &dmtemp
		if save {
			datamanagers_lock.Lock()
			datamanagers[m.ID] = dm
			datamanagers_lock.Unlock()
		}
	} else {
		dm2 := *dm
		dm2.UpdateInvaild()
	}

	return *dm
}
func DaysHistory() int {
	return 35
}

//This should die and is just to mak sure there is a URL() for the datamanger
type hasURL interface {
	URL() string
}

func GetDataManager(i interface{}) DataManager {
	if m, ok := i.(*catalog.Metric); ok {
		return GetDataManagerMetric(m)
	}
	if h, ok := i.(hasURL); ok {
		metrics := Rollup(i)
		alldata := GetDataManagerMetrics(metrics)
		return CombineMetricData(h.URL(), alldata)
	}
	return nil
}

func GetDataManagerMetrics(metrics []*catalog.Metric) []DataManager {
	allmanagers := []DataManager{}
	for _, m := range metrics {
		allmanagers = append(allmanagers, GetDataManagerMetric(m))
	}
	return allmanagers
}

func UnavailableMetrics(i interface{}, date time.Time) []*catalog.Metric {
	metrics := Rollup(i)
	return UnavailableMetricsFromMetricList(metrics, date)
}

func UnavailableMetricsFromMetricList(metrics []*catalog.Metric, date time.Time) []*catalog.Metric {
	unavailmetrics := []*catalog.Metric{}
	for _, m := range metrics {
		dayData := GetDataManagerMetric(m).Day(date)
		if dayData != nil {
			if dayData.Availability() < 100.0 {
				unavailmetrics = append(unavailmetrics, m)
			}
		}
	}
	return unavailmetrics
}

func UpdateMetricDataFeature(j *catalog.Feature) {
	//logrus.Infoln("UpdateData:Feature:", j.ID())
	for _, child := range j.Children {
		UpdateMetricDataFeature(child)
	}

	for _, metric := range j.Metrics {
		GetDataManagerMetric(metric).Update()
	}
}

func UpdateMetricData() {
	logrus.Warnln("UpdateReports:")
	if config.Config.Environment == "production" {
		jr := catalog.GetCatalog()
		if jr != nil {
			for _, metric := range jr.GetMetrics(nil) {
				logrus.Println("UpdateMetric:", metric.ID)
				GetDataManagerMetric(metric).Update()
			}
		}
	} else {
		logrus.Warnln("Skipping UpdateMetricData in non-production environment")
	}
}
