package datamanager

import (
	"encoding/json"
	"html/template"
	"time"

	"github.com/sirupsen/logrus"

	"code.justin.tv/availability/hms-esk/pkg/daydata"
)

// MetricDataInterval represents an arbitrary number of days of availability data
type MetricDataInterval interface {
	MetricID() uint
	URL() string
	MinutesDown() int
	MinutesTotal() int
	Availability() float64
	Latency(latencyType string) float64
	StartDate() time.Time
	DateStr() string
	DateRangeStr() string
	Days() []daydata.DayData
	AvailabilityBoolGraphData() template.JS
	AvailabilityBoolGraphDataDetailed() []daydata.GraphDatum
	AvailabilityPercentGraphData() template.JS
	AvailabilityPercentGraphDataPoints() []daydata.GraphDatum
	MinutesDownGraphData() template.JS
	MinutesDownGraphDataPoints() []daydata.GraphDatum
	MinutesDownBudgetGraphData() template.JS
	MinutesDownBudgetGraphDataPoints() []daydata.GraphDatum
	RawData() []*daydata.RawData
}

// MetricDataIntervalImpl returns mock data for an arbitrary week
type MetricDataIntervalImpl struct {
	metricData DataManager
	date       time.Time
	numDays    int
	days       []daydata.DayData
}

// NewMetricDataInterval returns metric data for the given interval
func NewMetricDataInterval(date time.Time, numDays int, metricData DataManager) MetricDataInterval {
	days := []daydata.DayData{}
	dayDate := date
	hasDays := false
	for i := 0; i < numDays; i++ {
		dayData := metricData.Day(dayDate) // FIXME: Deal with errors
		if dayData != nil {
			hasDays = true
		}
		// FIXME: Having nil days is bad, we should actually have days with
		// no data.
		days = append(days, dayData)
		dayDate = dayDate.Add(time.Duration(24) * time.Hour)
	}

	if !hasDays {
		logrus.Info("No days for MetricDataInterval: ", date, " numDays: ", numDays, " URL: ", metricData.URL())
		return nil
	}

	return &MetricDataIntervalImpl{
		metricData: metricData,
		date:       date,
		numDays:    numDays,
		days:       days,
	}
}

func (m *MetricDataIntervalImpl) MetricID() uint {
	return m.metricData.MetricID()
}

func (m *MetricDataIntervalImpl) URL() string {
	return m.metricData.URL()
}
func (m *MetricDataIntervalImpl) StartDate() time.Time {
	return m.date
}

func (m *MetricDataIntervalImpl) DateStr() string {
	return m.date.Format("2006-01-02")
}

func (m *MetricDataIntervalImpl) DateRangeStr() string {
	// Date range is inclusive so subtract one from the numDays (numDays should always be > 0?)
	return m.date.Format("2006-01-02") + " - " + m.date.Add(time.Duration(m.numDays-1)*24*time.Hour).Format("2006-01-02")
}

// MinutesDown returns the total minutes down over the entire interval
func (m *MetricDataIntervalImpl) MinutesDown() int {
	minutesDown := 0
	for _, day := range m.Days() {
		minutesDown += day.MinutesDown()
	}
	return minutesDown
}

// MinutesTotal returns the total minutes down over the entire interval
func (m *MetricDataIntervalImpl) MinutesTotal() int {
	minutesTotal := 0
	for _, day := range m.Days() {
		minutesTotal += day.MinutesTotal()
	}
	return minutesTotal
}

// Availability returns the availability over the entire interval.
func (m *MetricDataIntervalImpl) Availability() float64 {
	return AvailabilityPercentage(m.Days())
}

// Latency returns the average latency over the interval
func (m *MetricDataIntervalImpl) Latency(latencyType string) float64 {
	raw := m.RawData()
	sumLatency := 0.0
	sumIntervals := 0

	for _, rd := range raw {
		if rd.ID == latencyType {
			for _, p := range rd.Datapoints {
				sumLatency += p.Value
				sumIntervals++
			}
		}
	}
	if sumIntervals == 0 {
		return -1.0
	}
	return sumLatency / float64(sumIntervals)
}

// Days returns the availability data for the whole interval
func (m *MetricDataIntervalImpl) Days() []daydata.DayData {
	return m.days
}

// AvailabilityBoolGraphData returns graph data as booleans formatted for chart.js
func (m *MetricDataIntervalImpl) AvailabilityBoolGraphData() template.JS {
	gd := m.AvailabilityBoolGraphDataDetailed()
	bd, _ := json.Marshal(gd)
	return template.JS(string(bd))
}

// AvailabilityBoolGraphDataPoints returns graph data as booleans as points
func (m *MetricDataIntervalImpl) AvailabilityBoolGraphDataDetailed() []daydata.GraphDatum {
	gd := []daydata.GraphDatum{}

	for _, day := range m.Days() {
		if day != nil {
			// FIXME: nil days suck
			for _, dp := range day.AvailabilityData() {
				gd = append(gd, daydata.GraphDatum{dp.Time.Format(time.RFC3339), daydata.BoolToFloat64(dp.Available)})
			}
		}
	}

	return gd
}

// AvailabilityPercentGraphData returns graph data formatted for chart.js
func (m *MetricDataIntervalImpl) AvailabilityPercentGraphData() template.JS {
	gd := m.AvailabilityPercentGraphDataPoints()

	bd, _ := json.Marshal(gd)
	return template.JS(string(bd))
}

// AvailabilityPercentGraphDataPoints returns graph data as points
func (m *MetricDataIntervalImpl) AvailabilityPercentGraphDataPoints() []daydata.GraphDatum {
	gd := []daydata.GraphDatum{}
	for _, curDay := range m.days {
		if curDay != nil {
			gd = append(gd, daydata.GraphDatum{curDay.Date().Format(time.RFC3339), curDay.Availability()})
		}
	}
	return gd
}

func (m *MetricDataIntervalImpl) MinutesDownGraphData() template.JS {
	gd := m.MinutesDownGraphDataPoints()
	bd, _ := json.Marshal(gd)
	return template.JS(string(bd))
}

func (m *MetricDataIntervalImpl) MinutesDownGraphDataPoints() []daydata.GraphDatum {
	gd := []daydata.GraphDatum{}
	md := 0.0

	lastHour := -1
	for _, day := range m.Days() {
		if day != nil {
			// FIXME: nil days suck
			var lastDP daydata.AvailabilityDatum
			for _, dp := range day.AvailabilityData() {
				lastDP = dp
				if !dp.Available {
					md += 5.0 // FIXME: Don't hardcode this
				}
				if dp.Time.Hour() != lastHour {
					gd = append(gd, daydata.GraphDatum{dp.Time.Format(time.RFC3339), md})
					lastHour = dp.Time.Hour()
				}
			}
			gd = append(gd, daydata.GraphDatum{lastDP.Time.Format(time.RFC3339), md})
		}
	}
	return gd
}

func (m *MetricDataIntervalImpl) MinutesDownBudgetGraphData() template.JS {
	gd := m.MinutesDownBudgetGraphDataPoints()

	bd, _ := json.Marshal(gd)
	return template.JS(string(bd))
}

func (m *MetricDataIntervalImpl) MinutesDownBudgetGraphDataPoints() []daydata.GraphDatum {
	gd := []daydata.GraphDatum{}
	md := 0.0

	downtimePerMinute := 1.0 * 0.001
	lastHour := -1
	for _, day := range m.Days() {
		if day != nil {
			// FIXME: nil days suck
			var lastDP daydata.AvailabilityDatum
			for _, dp := range day.AvailabilityData() {
				lastDP = dp
				md += downtimePerMinute * 5
				if dp.Time.Hour() != lastHour {
					gd = append(gd, daydata.GraphDatum{dp.Time.Format(time.RFC3339), md})
					lastHour = dp.Time.Hour()
				}
			}
			gd = append(gd, daydata.GraphDatum{lastDP.Time.Format(time.RFC3339), md})
		}
	}
	return gd
}

func (m *MetricDataIntervalImpl) RawData() []*daydata.RawData {
	rdArray := []*daydata.RawData{}

	for _, day := range m.Days() {
		if day != nil {
			// FIXME: nil days suck

			dayRawData := day.RawData()
			for i, rawData := range dayRawData {
				if i >= len(rdArray) {
					rdArray = append(rdArray, daydata.NewRawData(rawData.ID, rawData.Description, rawData.Label, rawData.Unit))
				}
				rdArray[i].Append(rawData)
			}
		}
	}

	return rdArray
}
