package datamanager

import (
	"code.justin.tv/availability/goracle/catalog"
	"github.com/stretchr/testify/assert"
	"testing"
)

func testService(name string, components ...*catalog.Component) *catalog.Service {
	return &catalog.Service{
		Name:       name,
		Components: components,
	}
}

func testComponent(name string, rollup bool, metrics ...*catalog.Metric) *catalog.Component {
	return &catalog.Component{
		Name:    name,
		Rollup:  rollup,
		Metrics: metrics,
	}
}
func testFeature(name string, children []*catalog.Feature, metrics ...*catalog.Metric) *catalog.Feature {
	return &catalog.Feature{
		Name:     name,
		Children: children,
		Metrics:  metrics,
	}
}

func testMetric(name string, compRollup, featRollup bool) *catalog.Metric {
	return &catalog.Metric{
		Name:            name,
		ComponentRollup: compRollup,
		FeatureRollup:   featRollup}
}

func TestRollupComponentNoValidMetrics(t *testing.T) {
	comp := testComponent("comp0", false,
		testMetric("0", false, false),
		testMetric("1", false, false),
		testMetric("2", false, true),
		testMetric("3", false, true),
	)

	mets := Rollup(comp)
	assert.Len(t, mets, 0, "No metrics should be retuned")
}

func TestRollupComponentOneValidMetric(t *testing.T) {
	comp := testComponent("comp0", false,
		testMetric("0", false, false),
		testMetric("1", true, false),
	)

	mets := Rollup(comp)
	assert.Len(t, mets, 1, "One metric should be retuned")
	assert.Contains(t, mets, testMetric("1", true, false), "The list does not contain metric 1")
}

func TestRollupComponentMultipleValidMetric(t *testing.T) {
	comp := testComponent("comp0", false,
		testMetric("0", false, false),
		testMetric("1", true, false),
		testMetric("2", true, false),
		testMetric("3", true, true),
		testMetric("4", true, true),
	)

	mets := Rollup(comp)
	assert.Len(t, mets, 4, "Four metrics should be retuned")
	assert.Contains(t, mets, testMetric("1", true, false), "The list does not contain metric 1")
	assert.Contains(t, mets, testMetric("2", true, false), "The list does not contain metric 2")
	assert.Contains(t, mets, testMetric("3", true, true), "The list does not contain metric 3")
	assert.Contains(t, mets, testMetric("4", true, true), "The list does not contain metric 4")
	assert.NotContains(t, mets, testMetric("0", false, false), "The list contains metric 0")
}

func TestRollupServiceNoValidMetrics(t *testing.T) {
	serv := testService("serv1",
		testComponent("comp0", false, testMetric("0", false, false)),
		testComponent("comp1", false, testMetric("0", true, false)),
		testComponent("comp2", true, testMetric("0", false, false)),
	)

	mets := Rollup(serv)
	assert.Len(t, mets, 0, "No metrics should be retuned")
}

func TestRollupServiceOneValidMetrics(t *testing.T) {
	serv := testService("serv1",
		testComponent("comp0", true,
			testMetric("0", false, true),
			testMetric("1", false, false),
			testMetric("2", true, false)),
		testComponent("comp1", true, testMetric("3", false, false)),
	)

	mets := Rollup(serv)
	assert.Len(t, mets, 1, "One metric should be retuned")
	assert.Contains(t, mets, testMetric("2", true, false), "The list does not contain metric 1")
}

func TestRollupServiceMultipeValidMetrics(t *testing.T) {
	serv := testService("serv1",
		testComponent("comp0", true,
			testMetric("0", false, true),
			testMetric("1", false, false),
			testMetric("2", true, false)),
		testComponent("comp1", true, testMetric("3", true, false)),
	)

	mets := Rollup(serv)
	assert.Len(t, mets, 2, "Two metrics should be retuned")
	assert.Contains(t, mets, testMetric("2", true, false), "The list does not contain metric 2")
	assert.Contains(t, mets, testMetric("3", true, false), "The list does not contain metric 3")
}

func TestRollupMetrics(t *testing.T) {
	metrics := []*catalog.Metric{
		testMetric("0", false, false),
		testMetric("1", false, false),
		testMetric("2", true, false),
		testMetric("3", false, true),
		testMetric("4", true, true),
	}
	for _, m := range metrics {
		mets := Rollup(m)
		assert.Len(t, mets, 1, "One metric should be retuned")
		assert.Contains(t, mets, m, "The list does not contain metric %s", m.Name)
	}
}

type invalid struct{}

func TestInvalidStruct(t *testing.T) {
	mets := Rollup(invalid{})
	assert.Nil(t, mets, "No metrics should be retuned")
}

func TestRollupFeatureNoVaild(t *testing.T) {
	feat := testFeature("feat0",
		[]*catalog.Feature{testFeature("feat1", nil,
			testMetric("0", false, false),
			testMetric("1", true, false),
			testMetric("2", true, false))},
		testMetric("3", true, false),
	)
	mets := Rollup(feat)
	assert.Len(t, mets, 0, "No metrics should be retuned")
}
func TestRollupFeatureMultipeVaild(t *testing.T) {
	feat := testFeature("feat0",
		[]*catalog.Feature{testFeature("feat1", nil,
			testMetric("0", false, true),
			testMetric("1", true, false),
			testMetric("2", false, false))},
		testMetric("3", false, true),
	)
	mets := Rollup(feat)
	assert.Len(t, mets, 2, "Two metrics should be retuned")
	assert.Contains(t, mets, testMetric("0", false, true), "The list does not contain metric 0")
	assert.Contains(t, mets, testMetric("3", false, true), "The list does not contain metric 3")
}
