package datasource

import (
	"fmt"
	"time"

	"github.com/jinzhu/now"
	"github.com/sirupsen/logrus"

	"code.justin.tv/availability/goracle/catalog"
	"code.justin.tv/availability/hms-esk/pkg/daydata"
	"code.justin.tv/availability/hms-esk/pkg/graphitedata"
)

// GraphiteErrorDataSource implements a MetricDataSource which pulls availability
// metric data from graphite.
type GraphiteErrorDataSource struct {
	metricID  uint
	threshold float64
	url       string
	queries   []*catalog.Query
}

func NewGraphiteErrorDataSource(metricID uint, url string, queries []*catalog.Query, threshold float64) DataSource {

	if catalog.GetQueryOfType(queries, catalog.RequestCountQuery) == nil ||
		catalog.GetQueryOfType(queries, catalog.ErrorCountQuery) == nil {
		logrus.Debugf("error rate metric %d has no error rate and/or count query, returning NilDatasource", metricID)
		return &NilDatasource{
			metricID: metricID,
			url:      url,
		}
	}

	return &GraphiteErrorDataSource{
		metricID:  metricID,
		threshold: threshold,
		url:       url,
		queries:   queries,
	}
}

// Type returns the human readable type of the data source
func (g *GraphiteErrorDataSource) Type() string {
	return "GraphiteError"
}

// errorDayData returns a "default" DayData in the case of errors
func (g *GraphiteErrorDataSource) errorDayData(date time.Time) daydata.DayData {
	availabilityData := []daydata.AvailabilityDatum{}
	rawData := []*daydata.RawData{}

	return daydata.NewDayData(g.metricID, g.url, date, availabilityData, rawData, false)
}

// DayData fetches data for the 24 hour period starting at midnight of the
// provided date
func (g *GraphiteErrorDataSource) DayData(date time.Time) daydata.DayData {

	requestCountQuery := catalog.GetQueryOfType(g.queries, catalog.RequestCountQuery)
	errorCountQuery := catalog.GetQueryOfType(g.queries, catalog.ErrorCountQuery)

	requestGraphite, err := fetchDataForQuery(requestCountQuery, date)
	if err != nil {
		// logrus.Warnf("Metric %d: Graphite data fetch failed for %s", g.metricID, requestCountQuery.Query)
		return g.errorDayData(date)
	}
	errorGraphite, err := fetchDataForQuery(errorCountQuery, date)
	if err != nil {
		// logrus.Warnf("Metric %d: Graphite data fetch failed for %s", g.metricID, errorCountQuery.Query)
		return g.errorDayData(date)
	}

	return g.dayDataFromData(date, requestGraphite, errorGraphite)
}

// DayData fetches data for the 24 hour period starting at midnight of the
// provided date
func (g *GraphiteErrorDataSource) dayDataFromData(date time.Time, requestData, errorData *graphitedata.GraphiteData) daydata.DayData {
	midnight := now.New(date).BeginningOfDay()

	requestCountQuery := catalog.GetQueryOfType(g.queries, catalog.RequestCountQuery)
	errorCountQuery := catalog.GetQueryOfType(g.queries, catalog.ErrorCountQuery)

	requestCountRaw := daydata.NewRawDataFromGraphite("Request Count", requestCountQuery.Query, "errors", "count/sec", *requestData, daydata.GetAggregateFunction(requestCountQuery.AggregateType), midnight)
	errorCountRaw := daydata.NewRawDataFromGraphite("Error Count", errorCountQuery.Query, "requests", "count/sec", *errorData, daydata.GetAggregateFunction(errorCountQuery.AggregateType), midnight)
	errorRateRaw := rawErrorRate(*requestCountRaw, *errorCountRaw)
	allRawData := []*daydata.RawData{errorRateRaw, requestCountRaw, errorCountRaw}

	availabilityData := availabilityFromRawThreshold(*errorRateRaw, Below, g.threshold)

	return daydata.NewDayData(g.metricID, g.url, midnight, availabilityData, allRawData, true)
}

func (g *GraphiteErrorDataSource) DaysData(numDays int) map[time.Time]daydata.DayData {

	requestCountQuery := catalog.GetQueryOfType(g.queries, catalog.RequestCountQuery)
	errorCountQuery := catalog.GetQueryOfType(g.queries, catalog.ErrorCountQuery)
	end := now.BeginningOfDay()
	start := end.Add(time.Duration(-24*numDays) * time.Hour)

	requestData, _ := graphitedata.FetchDynamoDailyQueryHistoryRange(fmt.Sprint(requestCountQuery.ID), start, end)

	errorData, _ := graphitedata.FetchDynamoDailyQueryHistoryRange(fmt.Sprint(errorCountQuery.ID), start, end)
	days := map[time.Time]daydata.DayData{}
	for cur := start; !cur.After(end); cur = cur.Add(time.Duration(24) * time.Hour) {
		datestr := graphitedata.TimeToDateKey(cur)
		ed := errorData[datestr]
		rd := requestData[datestr]
		if ed == nil {
			days[cur] = g.errorDayData(cur)
			SendQueryToEcho(errorCountQuery, cur)
			logrus.Warnf("nil ed %d %d %v", g.metricID, errorCountQuery.ID, cur)
		} else if rd == nil {
			days[cur] = g.errorDayData(cur)
			logrus.Warnf("nil %d %v", g.metricID, cur)
			SendQueryToEcho(requestCountQuery, cur)
		} else {
			days[cur] = g.dayDataFromData(cur, rd, ed)
		}
	}

	return days
}
