package datasource

import (
	"fmt"
	"time"

	"code.justin.tv/availability/goracle/catalog"
	"code.justin.tv/availability/hms-esk/pkg/daydata"
	"code.justin.tv/availability/hms-esk/pkg/graphitedata"

	"github.com/jinzhu/now"
	"github.com/sirupsen/logrus"
)

// GraphiteLatencyDataSource implements a DataSource which collects latency data
// from graphite - in particular, p50, p90, and p99.
type GraphiteLatencyDataSource struct {
	metricID              uint
	url                   string
	queries               []*catalog.Query
	availabilityQueryType catalog.QueryType
	threshold             float64
}

func (g *GraphiteLatencyDataSource) Type() string {
	return "GraphiteLatency"
}

func NewGraphiteLatencyDataSource(metricID uint, url string, availabilityQueryType catalog.QueryType, threshold float64, queries []*catalog.Query) DataSource {
	query := catalog.GetQueryOfType(queries, availabilityQueryType)
	if query == nil {
		logrus.Debugf("Latency metric %d has no query for query type %s, returning NilDatasource", metricID, availabilityQueryType)
		return &NilDatasource{
			metricID: metricID,
			url:      url,
		}
	}

	return &GraphiteLatencyDataSource{
		metricID: metricID,
		url:      url,
		availabilityQueryType: availabilityQueryType,
		threshold:             threshold,
		queries:               queries,
	}
}

// errorDayData returns a "default" DayData in the case of errors
func (g *GraphiteLatencyDataSource) errorDayData(date time.Time) daydata.DayData {
	availabilityData := []daydata.AvailabilityDatum{}
	rawData := []*daydata.RawData{}

	return daydata.NewDayData(g.metricID, g.url, date, availabilityData, rawData, false)
}

// DayData fetches data for the 24 hour period starting at midnight of the
// provided date
func (g *GraphiteLatencyDataSource) DayData(date time.Time) daydata.DayData {
	midnight := now.New(date).BeginningOfDay()

	allRawData := []*daydata.RawData{}
	availabilityData := []daydata.AvailabilityDatum{}
	for _, q := range g.queries {
		graphiteData, err := fetchDataForQuery(q, date)
		if err != nil {
			logrus.Debugf("Metric %d: Graphite data fetch failed for %s", g.metricID, q.Query)
			return g.errorDayData(date)
		}

		rawData := daydata.NewRawDataFromGraphite(string(q.QueryType), q.Query, "request latency", "msecs", *graphiteData, daydata.GetAggregateFunction(q.AggregateType), midnight)
		if q.QueryType == g.availabilityQueryType {
			availabilityData = availabilityFromRawThreshold(*rawData, Below, g.threshold)
		}
		allRawData = append(allRawData, rawData)
	}

	return daydata.NewDayData(g.metricID, g.url, midnight, availabilityData, allRawData, true)
}

func (g *GraphiteLatencyDataSource) DaysData(numDays int) map[time.Time]daydata.DayData {
	end := now.BeginningOfDay()
	start := end.Add(time.Duration(-24*numDays) * time.Hour)
	datas := map[uint]map[string]*graphitedata.GraphiteData{}
	for _, q := range g.queries {
		data, _ := graphitedata.FetchDynamoDailyQueryHistoryRange(fmt.Sprint(q.ID), start, end)
		datas[q.ID] = data
	}

	days := map[time.Time]daydata.DayData{}
	for cur := start; !cur.After(end); cur = cur.Add(time.Duration(24) * time.Hour) {
		datestr := graphitedata.TimeToDateKey(cur)
		allRawData := []*daydata.RawData{}
		availabilityData := []daydata.AvailabilityDatum{}
		for _, q := range g.queries {
			data := datas[q.ID][datestr]
			if data == nil {
				SendQueryToEcho(q, cur)
				days[cur] = g.errorDayData(cur)
			} else {
				rawData := daydata.NewRawDataFromGraphite(string(q.QueryType), q.Query, "request latency", "msecs", *data, daydata.GetAggregateFunction(q.AggregateType), cur)
				if q.QueryType == g.availabilityQueryType {
					availabilityData = availabilityFromRawThreshold(*rawData, Below, g.threshold)
				}
				allRawData = append(allRawData, rawData)
			}
		}
		_, exists := days[cur]
		if !exists {
			days[cur] = daydata.NewDayData(g.metricID, g.url, cur, availabilityData, allRawData, true)
		}
	}

	return days
}
