package datasource

import (
	"fmt"
	"time"

	"github.com/sirupsen/logrus"

	"github.com/jinzhu/now"

	"code.justin.tv/availability/goracle/catalog"
	"code.justin.tv/availability/hms-esk/pkg/daydata"
	"code.justin.tv/availability/hms-esk/pkg/graphitedata"
)

// GraphiteThresholdDataSource implements a MetricDataSource which pulls availability
// metric data from graphite.
type GraphiteThresholdDataSource struct {
	metricID      uint
	url           string
	query         *catalog.Query
	threshold     float64
	thresholdType ThresholdType
	aggregateType catalog.AggregateType
	thresholdFunc ThresholdFunc
	aggregateFunc daydata.AggregateFunc
	params        map[string]interface{}
}

// NewGraphiteThresholdDataSource instantiates a GraphiteDataSource from standard catalog.Metric parameters
func NewGraphiteThresholdDataSource(metricID uint, url string, queries []*catalog.Query, threshold float64, thresholdType ThresholdType, thresholdFunc ThresholdFunc) DataSource {
	query := catalog.GetQueryOfType(queries, catalog.ThresholdQuery)
	if query == nil {
		logrus.Warnf("threshold metric %d has no threshold type query, returning NilDatasource", metricID)
		return &NilDatasource{
			metricID: metricID,
			url:      url,
		}
	}
	aggregateType := query.AggregateType
	aggregateFunc := daydata.GetAggregateFunction(aggregateType)

	return &GraphiteThresholdDataSource{
		metricID:      metricID,
		url:           url,
		query:         query,
		threshold:     threshold,
		thresholdType: thresholdType,
		thresholdFunc: thresholdFunc,
		aggregateType: aggregateType,
		aggregateFunc: aggregateFunc,
	}
}

// Type returns a human description of the data source type
func (g *GraphiteThresholdDataSource) Type() string {
	return "GraphiteThreshold"
}

func (g *GraphiteThresholdDataSource) errorDayData(date time.Time) daydata.DayData {
	availabilityData := []daydata.AvailabilityDatum{}
	rawData := []*daydata.RawData{}

	return daydata.NewDayData(g.metricID, g.url, date, availabilityData, rawData, false)
}

// DayData fetches data for the 24 hour period starting at midnight of the
// provided date
func (g *GraphiteThresholdDataSource) DayData(date time.Time) daydata.DayData {

	graphiteData, err := fetchDataForQuery(g.query, date)
	if err != nil {
		logrus.Debugf("Metric %d: Graphite data fetch failed for %s", g.metricID, g.query.Query)
		return g.errorDayData(date)
	}
	return g.dayDataFromData(date, graphiteData)
}

// DayData fetches data for the 24 hour period starting at midnight of the
// provided date
func (g *GraphiteThresholdDataSource) dayDataFromData(date time.Time, data *graphitedata.GraphiteData) daydata.DayData {
	midnight := now.New(date).BeginningOfDay()

	rawData := daydata.NewRawDataFromGraphite("threshold query", g.query.Query, "", "", *data, g.aggregateFunc, midnight)
	availabilityData := availabilityFromRawThreshold(*rawData, g.thresholdFunc, g.threshold)
	allRawData := []*daydata.RawData{rawData}

	return daydata.NewDayData(g.metricID, g.url, midnight, availabilityData, allRawData, true)
}

func (g *GraphiteThresholdDataSource) DaysData(numDays int) map[time.Time]daydata.DayData {
	end := now.BeginningOfDay()
	start := end.Add(time.Duration(-24*numDays) * time.Hour)

	data, _ := graphitedata.FetchDynamoDailyQueryHistoryRange(fmt.Sprint(g.query.ID), start, end)

	days := map[time.Time]daydata.DayData{}

	for cur := start; !cur.After(end); cur = cur.Add(time.Duration(24) * time.Hour) {
		datestr := graphitedata.TimeToDateKey(cur)
		d := data[datestr]
		if d == nil {
			days[cur] = g.errorDayData(cur)
			SendQueryToEcho(g.query, cur)
			logrus.Warnf("nil data %d %d %v", g.metricID, g.query.ID, cur)
		} else {
			days[cur] = g.dayDataFromData(cur, d)
		}
	}

	return days
}
