package datasource

import (
	"encoding/json"
	"fmt"
	"time"

	"code.justin.tv/availability/goracle/catalog"
	"code.justin.tv/availability/hms-esk/pkg/config"
	"code.justin.tv/availability/hms-esk/pkg/daydata"
	"code.justin.tv/availability/hms-esk/pkg/graphitedata"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/jinzhu/now"
	"github.com/sirupsen/logrus"
)

// ThresholdFunc takes a value and a threshold as arguments and returns true if it passes
type ThresholdFunc func(value float64, threshold float64) bool

// ThresholdType is the string that describs threshold types
type ThresholdType string

const (
	ABOVE ThresholdType = "above"
	BELOW               = "below"
)

// Above returns true if the value is above the threshold
func Above(value float64, threshold float64) bool {
	return value > threshold
}

// Below returns true if the value is below the threshold
func Below(value float64, threshold float64) bool {
	return value < threshold
}

// fetchDataForQuery fetches queries from dynamo.
func fetchDataForQuery(query *catalog.Query, date time.Time) (*graphitedata.GraphiteData, error) {
	midnight := now.New(date).BeginningOfDay()

	// FIXME: Improve the quality of graphite response handling to generate more useful errors.
	// In particular, deal with missing aggregations.

	returnData, err := graphitedata.FetchDynamoDailyQueryHistory(fmt.Sprint(query.ID), midnight)

	logrus.Debugln("Query:", query.Query)

	if err != nil {
		logrus.Debugln("Dynamo query error:", err)
		errsqs := SendQueryToEcho(query, date)
		if errsqs != nil {
			logrus.Warnln("Echo query error:", errsqs)
		}
		return nil, err
	}

	return returnData, nil
}

type echoRequest struct {
	QueryID       uint
	Query         string
	AggregateType string
	Date          time.Time
}

func SendQueryToEcho(query *catalog.Query, date time.Time) error {
	awscfg := aws.NewConfig().WithRegion(config.Config.Region)
	if config.Config.SqsEndpoint != "" {
		awscfg.WithEndpoint(config.Config.SqsEndpoint)
	}
	svc := sqs.New(session.New(), awscfg)
	if query.ID == 0 {
		logrus.Error("esk is trying to send a zero query")
		return fmt.Errorf("esk is trying to send a zero query")
	}
	r := &echoRequest{
		QueryID:       query.ID,
		Query:         query.Query,
		AggregateType: string(query.AggregateType),
		Date:          date,
	}
	msg, err := json.Marshal(r)
	if err != nil {
		return err
	}
	msgstr := string(msg)
	qurl := config.QueueURL()
	in := &sqs.SendMessageInput{
		MessageBody: &msgstr,
		QueueUrl:    &qurl,
	}
	_, err = svc.SendMessage(in)

	if err != nil {
		logrus.Warn(err)
		return err
	}
	return nil
}

// fetchDataForQuery fetches queries from dynamo.
func FetchDataForQuery(query *catalog.Query, date time.Time) (*graphitedata.GraphiteData, error) {
	return fetchDataForQuery(query, date)
}

func availabilityFromRawThreshold(rawData daydata.RawData, tf ThresholdFunc, threshold float64) []daydata.AvailabilityDatum {
	avData := []daydata.AvailabilityDatum{}
	for _, p := range rawData.Datapoints {
		available := false
		if tf(p.Value, threshold) {
			available = true
		}
		avData = append(avData, daydata.AvailabilityDatum{
			Time:      p.Time,
			Available: available,
		})
	}
	return avData
}

func rawErrorRate(rData daydata.RawData, eData daydata.RawData) *daydata.RawData {
	rawData := daydata.NewRawData("Error Rate", "Calculated Error Rate", "Error Rate", "% Errors")

	eLen := len(eData.Datapoints)
	rLen := len(rData.Datapoints)

	// Iterate through all time buckets represented in errors and requests
	// Error rate is:
	// error count/request count if there is both data
	// zero if there is request data but not error data (no errors reported, or unknown errors)
	// skipped if there is error data but not request data (divide by ???)
	eOffset := 0
	rOffset := 0
	for rOffset < rLen {
		rPoint := rData.Datapoints[rOffset]
		rTime := rPoint.Time
		if eOffset >= eLen {
			// No remaining error data, just add a zero error point
			rawData.Datapoints = append(rawData.Datapoints, daydata.FloatDatum{
				Time:  rTime,
				Value: 0.0,
			})
			rOffset++
		} else {
			ePoint := eData.Datapoints[eOffset]
			eTime := ePoint.Time

			if eTime.Before(rTime) {
				eOffset++
			} else if eTime.After(rTime) {
				// Request data, but no error data. Create a new point with a zero error rate.
				rawData.Datapoints = append(rawData.Datapoints, daydata.FloatDatum{
					Time:  rTime,
					Value: 0.0,
				})
				rOffset++
			} else {
				if rPoint.Value != 0.0 {
					// If request count is zero and error count is not, this is obviously not correct data
					// and we should skip.
					errorRate := float64(ePoint.Value) / rPoint.Value // Avoid divide by zero
					rawData.Datapoints = append(rawData.Datapoints, daydata.FloatDatum{
						Time:  rTime,
						Value: 100.0 * errorRate,
					})
				}
				eOffset++
				rOffset++
			}
		}
	}
	return rawData
}
