package datasource

import (
	"testing"
	"time"

	"code.justin.tv/availability/hms-esk/pkg/daydata"

	"github.com/stretchr/testify/assert"
)

func sequentialTimes(count int) []time.Time {
	t := time.Now()

	times := []time.Time{}

	for i := 0; i < count; i++ {
		times = append(times, t.Add(time.Duration(i)*daydata.BUCKET_DURATION))
	}
	return times
}

func TestAvailablityFromRawThreshold(t *testing.T) {
	assert := assert.New(t)

	rawData := daydata.RawData{}
	curTime := time.Now()
	rawData.AddPoint(curTime, 1.0)
	rawData.AddPoint(curTime, 7.0)
	adAbove := availabilityFromRawThreshold(rawData, Above, 5)
	assert.Equal(adAbove[0].Available, false, "1 < 5")
	assert.Equal(adAbove[1].Available, true, "7 > 5")
	adBelow := availabilityFromRawThreshold(rawData, Below, 5)
	assert.Equal(adBelow[0].Available, true, "1 < 5")
	assert.Equal(adBelow[1].Available, false, "7 > 5")
}

func TestRawErrorRate(t *testing.T) {
	assert := assert.New(t)

	reqData := daydata.RawData{}
	errData := daydata.RawData{}
	times := sequentialTimes(8)

	// No errors (zero) for time interval 0
	reqData.AddPoint(times[0], 100)
	errData.AddPoint(times[0], 0)

	// 5% error rate for time interval 1
	reqData.AddPoint(times[1], 100)
	errData.AddPoint(times[1], 5)

	// Missing error data for time interval 2
	reqData.AddPoint(times[2], 100)

	// Missing request data for time interval 3
	errData.AddPoint(times[3], 5)

	// Errors for time interval 4
	reqData.AddPoint(times[4], 100)
	errData.AddPoint(times[4], 10)

	// No errors for time interval 5
	reqData.AddPoint(times[5], 100)
	rateData := rawErrorRate(reqData, errData)

	assert.Equal(rateData.Datapoints[0].Value, 0.0, "error rate should be zero for first point")
	assert.Equal(rateData.Datapoints[1].Value, 5.0, "error rate should be 5%")
	assert.Equal(rateData.Datapoints[2].Value, 0.0, "error rate should be 0 with no error data")
	assert.Equal(rateData.Datapoints[3].Value, 10.0, "error rate skips points with no request data")
	assert.Equal(rateData.Datapoints[4].Value, 0.0, "error rate should exist even if no more error data exists")
	assert.Equal(len(rateData.Datapoints), 5, "error rate has the right number of data points")
}
