package daydata

import (
	"encoding/json"
	"html/template"
	"time"

	"github.com/jinzhu/now"
)

// AvailabilityDatum represents availability for a 5 minute window, starting
// at the specified time
type AvailabilityDatum struct {
	Time      time.Time
	Available bool
}

// FloatDatum represents a value for a particular point in time.
type FloatDatum struct {
	Time  time.Time `json:"time"`
	Value float64   `json:"value"`
}

// GraphDatum represent a value for use in chart.js graph data
type GraphDatum struct {
	X string  `json:"x"`
	Y float64 `json:"y"`
}

// DayData contains detailed data (5 minute interval) data about
// availability for a metric for a single day
type DayData interface {
	MetricID() uint
	URL() string
	Date() time.Time
	DateStr() string
	HasData() bool
	AvailabilityData() []AvailabilityDatum
	AvailabilityBoolGraphData() template.JS
	AvailabilityBoolGraphDataPoints() []GraphDatum
	MinutesDownGraphData() template.JS
	RawData() []*RawData
	MinutesDown() int
	MinutesTotal() int
	Availability() float64
	RecentAvailability() bool
	IsValidDay() bool
}

// DayDataImpl implements the AvailabilityDayDetail interface
type DayDataImpl struct {
	metricID         uint
	url              string
	date             time.Time
	availabilityData []AvailabilityDatum
	rawData          []*RawData
	availability     float64
	valid            bool
}

func NewDayData(metricID uint, url string, date time.Time, availabilityData []AvailabilityDatum, rawData []*RawData, valid bool) DayData {
	midnight := now.New(date).BeginningOfDay()
	return &DayDataImpl{
		metricID:         metricID,
		url:              url,
		date:             midnight,
		availabilityData: availabilityData,
		rawData:          rawData,
		valid:            valid,
	}
}

// MetricID returns the metric data that this
func (m *DayDataImpl) MetricID() uint {
	return m.metricID
}

func (m *DayDataImpl) IsValidDay() bool {
	return m.valid
}

func (m *DayDataImpl) URL() string {
	return m.url
}

// Date returns the date of this metric data
func (m *DayDataImpl) Date() time.Time {
	return m.date
}

func (m *DayDataImpl) DateStr() string {
	return m.date.Format("2006-01-02")
}

// HasData returns true if we have availability data for that
// full day.
func (m *DayDataImpl) HasData() bool {
	return len(m.availabilityData) > 0
}

// AvailabilityData returns all of the availability data for a particular day, sorted
// by time.
func (m *DayDataImpl) AvailabilityData() []AvailabilityDatum {
	return m.availabilityData
}

func BoolToFloat64(b bool) float64 {
	if b {
		return 1.0
	}
	return 0.0
}

// AvailabilityBoolGraphData returns graph data formatted for chart.js
func (m *DayDataImpl) AvailabilityBoolGraphData() template.JS {
	gd := m.AvailabilityBoolGraphDataPoints()
	bd, _ := json.Marshal(gd)
	return template.JS(string(bd))
}

// availabilityData.Points = &days returns graph data as points
func (m *DayDataImpl) AvailabilityBoolGraphDataPoints() []GraphDatum {
	gd := []GraphDatum{}
	for _, dp := range m.availabilityData {
		gd = append(gd, GraphDatum{dp.Time.Format(time.RFC3339), BoolToFloat64(dp.Available)})
	}
	return gd
}

// MinutesDownGraphData returns graph data formatted for chart.js
func (m *DayDataImpl) MinutesDownGraphData() template.JS {
	gd := []GraphDatum{}
	md := 0.0
	for _, dp := range m.availabilityData {
		if !dp.Available {
			md += 5.0 // FIXME: Don't hardcode this
		}
		gd = append(gd, GraphDatum{dp.Time.Format(time.RFC3339), md})
	}

	bd, _ := json.Marshal(gd)
	return template.JS(string(bd))
}

func (m *DayDataImpl) RawData() []*RawData {
	return m.rawData
}

func (m *DayDataImpl) MinutesDown() int {
	minutesDown := 0
	for _, point := range m.availabilityData {
		if !point.Available {
			minutesDown += 5 // FIXME: Don't hardcode this
		}
	}
	return minutesDown
}

func (m *DayDataImpl) MinutesTotal() int {
	// FIXME: don't hardcode this
	return 5 * len(m.availabilityData)
}

// Availability returns the availability across all time buckets for the
// day
func (m *DayDataImpl) Availability() float64 {
	if m.availability != 0.0 {
		return m.availability
	}
	sum := 0
	total := len(m.availabilityData)
	if total == 0 {
		return -1.0
	}
	for _, point := range m.availabilityData {
		if point.Available {
			sum += 1
		}
	}
	m.availability = 100.0 * (float64(sum) / float64(total))
	return m.availability
}

func (d *DayDataImpl) RecentAvailability() bool {
	// TODO: It is weird to do this for a day which isn't the current.
	if len(d.AvailabilityData()) == 0 {
		return true
	}
	return d.AvailabilityData()[len(d.AvailabilityData())-1].Available
}

// CombineMetricDataDay takes MetricDataDays and combines them to
// form a set of overall availability data for that day
func CombineMetricDataDay(url string, mdd []DayData) DayData {
	// FIXME: Assumes that all days have datapoints for the entire day
	if len(mdd) == 0 {
		return nil
	}
	midnight := mdd[0].Date()

	availabilityData := []AvailabilityDatum{}
	// 5 minute intervals for the entire day
	// FIXME: There is a fencepost error in the data source, this keeps the
	// numbers consistent.
	for i := 0; i <= 24*12; i++ {
		add := false
		available := true
		for _, dd := range mdd {
			if len(dd.AvailabilityData()) > i {
				add = true

				if !dd.AvailabilityData()[i].Available {
					available = false
				}
			}
		}
		if add {
			dp := AvailabilityDatum{
				Time:      midnight.Add(time.Duration(i*300) * time.Second),
				Available: available,
			}
			availabilityData = append(availabilityData, dp)
		}
	}
	return &DayDataImpl{
		metricID:         0,
		url:              url,
		date:             midnight,
		availabilityData: availabilityData,
	}
}
