package daydata

import (
	"encoding/json"
	"html/template"
	"time"

	"code.justin.tv/availability/hms-esk/pkg/graphitedata"

	"github.com/jinzhu/now"
)

// RawData represents minimally processed floating point metric data. RawData
// has usually only been aggregated to match time intervals. RawData also has
// no nil data. There can also be gaps in time within a RawData.
type RawData struct {
	ID          string
	Description string
	Label       string
	Unit        string
	Datapoints  []FloatDatum
}

func addBucket(dayStart time.Time, dayEnd time.Time, af AggregateFunc, raw *RawData, t time.Time, values []*float64) {
	if t.After(dayEnd) || t.Before(dayStart) {
		return
	}

	// Perform aggregation on points in the bucket.
	bucketVal := af(values)
	if bucketVal != nil {
		raw.AddPoint(t, *bucketVal)
	}
}

// NewRawData returns an empty RawData from parameters.
func NewRawData(id string, description string, label string, unit string) *RawData {
	return &RawData{
		ID:          id,
		Description: description,
		Datapoints:  []FloatDatum{},
		Label:       label,
		Unit:        unit,
	}
}

// NewRawDataFromGraphite takes graphite data, and a date. It
// time buckets raw data into 5 minute chunks using the provided aggregation function.
// This assumes that data coming in from graphite is valid.
// If a time interval has no data points, there will be no data point for that time interval.
func NewRawDataFromGraphite(id string, description string, label string, unit string, gd graphitedata.GraphiteData, af AggregateFunc, date time.Time) *RawData {
	raw := &RawData{
		ID:          id,
		Description: description,
		Datapoints:  []FloatDatum{},
		Label:       label,
		Unit:        unit,
	}

	dayBegin := now.New(date).BeginningOfDay()
	dayEnd := now.New(date).EndOfDay()

	numPoints := len(gd.Data)
	if numPoints == 0 {
		// FIXME: Should we return a RawData if there are no data points?
		return raw
	}

	values := []*float64{}

	// FIXME: Buckets are currently timestamped with start of the bucket, not end.
	curBucketStart := getBucketStart(date)
	for i := 0; i < numPoints; i++ {
		point := gd.Data[i]
		t := time.Unix(int64(*point.Timestamp), 0)

		bucketStart := getBucketStart(t)
		if !bucketStart.Equal(curBucketStart) {
			addBucket(dayBegin, dayEnd, af, raw, curBucketStart, values)

			// This datapoint is in a new bucket. Clear
			// information and restart
			curBucketStart = bucketStart
			values = []*float64{}
		}
		values = append(values, point.Value)
	}
	// Handle data in the last bucket.
	addBucket(dayBegin, dayEnd, af, raw, curBucketStart, values)

	return raw
}

// GraphData returns graph data formatted for chart.js
func (rd *RawData) GraphData() template.JS {
	gd := rd.GraphDataPoints()
	bd, _ := json.Marshal(gd)
	return template.JS(string(bd))
}

// GraphData returns graph data formatted for chart.js
func (rd *RawData) GraphDataPoints() []GraphDatum {
	gd := []GraphDatum{}
	for _, dp := range rd.Datapoints {
		gd = append(gd, GraphDatum{dp.Time.Format(time.RFC3339), dp.Value})
	}
	return gd
}

// AddPoint adds a datapoint to the raw data.
func (rd *RawData) AddPoint(time time.Time, value float64) {
	rd.Datapoints = append(rd.Datapoints, FloatDatum{time, value})
}

// Append adds all of the datapoints from one
func (rd *RawData) Append(source *RawData) {
	rd.Datapoints = append(rd.Datapoints, source.Datapoints...)
}
