package daydata

import (
	"math"
	"time"

	"github.com/sirupsen/logrus"

	"code.justin.tv/availability/goracle/catalog"
)

const BUCKET_DURATION = time.Duration(5) * time.Minute

// getBucketStart gets the start time of the bucket
func getBucketStart(t time.Time) time.Time {
	return t.Truncate(BUCKET_DURATION)
}

func getBucketEnd(t time.Time) time.Time {
	return t.Truncate(BUCKET_DURATION).Add(BUCKET_DURATION)
}

// AggregateFunc is a function that combines multiple float (or nil)
// data points and returns a resulting float or nil.
type AggregateFunc func([]*float64) *float64

// GetAggregateFunction returns the proper function to use for an aggregate type
func GetAggregateFunction(aggregateType catalog.AggregateType) AggregateFunc {

	switch aggregateType {
	case catalog.Sum:
		return SumAggregate
	case catalog.Average:
		return AverageAggregate
	case catalog.Max:
		return MaxAggregate
	default:
		logrus.Info("Invalid aggregate type:", aggregateType)
		return AverageAggregate
	}

}

// SumAggregate is an aggregator that simply sums all data points and returns the result, unless
// there are no non-nil data points, in which case it returns nil.
func SumAggregate(values []*float64) *float64 {
	count := 0
	sum := 0.0
	for _, v := range values {
		if v != nil {
			sum += *v
			count++
		}
	}
	if count > 0 {
		return &sum
	}

	return nil
}

// AverageAggregate is an aggregator that simply sums all data points and returns the result, unless
func AverageAggregate(values []*float64) *float64 {
	sum := 0.0
	count := 0
	for _, v := range values {
		if v != nil {
			sum += *v
			count++
		}
	}
	if count > 0 {
		average := sum / float64(count)
		return &average
	}

	return nil
}

// MaxAggregate is an aggregator that simply max all data points and returns the result, unless
// there are no non-nil data points, in which case it returns nil.
func MaxAggregate(values []*float64) *float64 {
	count := 0
	var max float64
	for _, v := range values {
		if v != nil {
			if count == 0 {
				max = *v
			} else {
				max = math.Max(max, *v)
			}
			count++
		}
	}
	if count > 0 {
		return &max
	}

	return nil
}
