package graphitedata

import (
	"encoding/json"
	"errors"
	"fmt"
	"time"

	"code.justin.tv/availability/goracle/catalog"
	"github.com/sirupsen/logrus"
)

// FetchGraphiteQueryData retries the graphite data for a query for a time period from graphite
func FetchGraphiteQueryData(query string, from, until time.Time) (*GraphiteData, error) {
	metricData, err := Render(query, from.Unix(), until.Unix())
	if err != nil {
		return nil, fmt.Errorf("Malformed query: %s", query)
	}
	if len(metricData) == 0 {
		return nil, fmt.Errorf("No data returned by query: %s", query)
	}
	if len(metricData) > 1 {
		return nil, fmt.Errorf("More then one series was returned by query: %s", query)
	}
	return metricData[0], nil
}

// FetchGraphiteAvailabilityData retries the graphite data for a list of
// queries of the old requestCount errorCount errorRate format and returns
// the data for them in that order.
func FetchGraphiteAvailabilityData(queries []*catalog.Query, from, until time.Time) ([]*GraphiteData, error) {
	if queries == nil {
		return nil, errors.New("Can not fetch graphite data for an empty list of queries")
	}
	requestCount := catalog.GetQueryStringOfType(queries, catalog.RequestCountQuery)
	requestCountData, err := Render(requestCount, from.Unix(), until.Unix())
	if err != nil {
		return nil, err
	}
	// Tag the data with the name "RequestCount"
	if len(requestCountData) > 0 {
		requestCountData[0].Name = "RequestCount"
	}
	errorCount := catalog.GetQueryStringOfType(queries, catalog.ErrorCountQuery)
	errorCountData, err := Render(errorCount, from.Unix(), until.Unix())
	if err != nil {
		return nil, err
	}
	// because weird things happen...
	if len(errorCountData) > 0 {
		errorCountData[0].Name = "ErrorCount"
	}
	errorRate := catalog.GetQueryStringOfType(queries, catalog.ErrorRateQuery)
	errorRateData, err := Render(errorRate, from.Unix(), until.Unix())
	if err != nil {
		return nil, err
	}
	if len(errorRateData) > 0 {
		errorRateData[0].Name = "ErrorRate"
	}
	results := []*GraphiteData{}
	results = append(results, requestCountData...)
	results = append(results, errorCountData...)
	results = append(results, errorRateData...)

	logrus.Debugf("RequestCount Query: %s", requestCount)
	logrus.Debugf("ErrorCount Query: %s", errorCount)
	logrus.Debugf("ErrorRate Query: %s", errorRate)

	res, err := json.Marshal(results)
	if err != nil {
		logrus.Warn("JSON marshalling error:", err)
		return nil, err
	}

	logrus.Debugf("Consolidated Query Results: %s", res)
	return results, nil
}
