package graphitedata

import (
	"encoding/json"
	"errors"
	"time"

	"github.com/sirupsen/logrus"

	c "code.justin.tv/availability/hms-esk/pkg/config"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
)

// FetchDynamoDailyQueryHistory attempts to fetch the data for a query id for a date from the GoracleQueryHistory dynamo table
func FetchDynamoDailyQueryHistory(queryID string, date time.Time) (*GraphiteData, error) {
	conf := &aws.Config{Region: aws.String("us-west-2"), Endpoint: aws.String(c.Config.DynamoDBEndpoint)}
	dynamo := dynamodb.New(session.New(), conf.WithMaxRetries(10))

	params := &dynamodb.GetItemInput{
		TableName: aws.String("GoracleQueryHistory"),
		Key: map[string]*dynamodb.AttributeValue{
			"QueryID": {
				S: aws.String(queryID),
			},
			"Date": {
				S: aws.String(TimeToDateKey(date)),
			},
		},
	}
	var resp *dynamodb.GetItemOutput
	var err error
	if resp, err = dynamo.GetItem(params); err != nil {
		// fmt.Println(resp)
		logrus.Warn("Dynamo Error:", err.Error())
		return nil, err
	}

	if resp.Item["Data"] == nil {
		logrus.Debugln("No Data in DynamoDailyQueryHistory: ", queryID, TimeToDateKey(date))
		return nil, errors.New("No Data in DynamoDailyQueryHistory")
	}
	rawData := resp.Item["Data"].S
	logrus.Debugln("Data in DynamoDailyQueryHistory: ", queryID, date)

	return parseGraphiteData(rawData)
}

// FetchDynamoDailyQueryHistory attempts to fetch the data for a query id for a date from the GoracleQueryHistory dynamo table
func FetchDynamoDailyQueryHistoryRange(queryID string, start, end time.Time) (map[string]*GraphiteData, error) {
	keys := []map[string]*dynamodb.AttributeValue{}
	for cur := start; !cur.After(end); cur = cur.Add(time.Duration(24) * time.Hour) {
		keys = append(keys, map[string]*dynamodb.AttributeValue{
			"QueryID": {
				S: aws.String(queryID),
			},
			"Date": {
				S: aws.String(TimeToDateKey(cur)),
			},
		})
	}
	conf := &aws.Config{Region: aws.String("us-west-2"), Endpoint: aws.String(c.Config.DynamoDBEndpoint)}
	dynamo := dynamodb.New(session.New(), conf.WithMaxRetries(10))

	kanda := dynamodb.KeysAndAttributes{
		Keys: keys,
	}
	params := &dynamodb.BatchGetItemInput{
		RequestItems: map[string]*dynamodb.KeysAndAttributes{"GoracleQueryHistory": &kanda},
	}
	var resp *dynamodb.BatchGetItemOutput
	var err error
	if resp, err = dynamo.BatchGetItem(params); err != nil {
		// fmt.Println(resp)
		logrus.Warn("Dynamo Error:", err.Error())
		return nil, err
	}
	daysdata := map[string]*GraphiteData{}
	for _, item := range resp.Responses["GoracleQueryHistory"] {
		if item["Data"] != nil {
			rawData := item["Data"].S
			data, _ := parseGraphiteData(rawData)
			daysdata[*item["Date"].S] = data
		}
	}
	return daysdata, nil
}

func parseGraphiteData(rawData *string) (*GraphiteData, error) {
	var parsedData *GraphiteData
	if err := json.Unmarshal([]byte(*rawData), &parsedData); err != nil {
		logrus.Debugf(err.Error())
		return nil, err
	}
	return parsedData, nil
}

// Will take a time.Time object and map it to the `Date` key value
// that a report for that day will be stored as in DynamoDB
func TimeToDateKey(t time.Time) string {
	return t.Format("2006-01-02")
}
