package reportdata

import (
	"time"

	c "code.justin.tv/availability/hms-esk/pkg/config"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/guregu/dynamo"
	"github.com/sirupsen/logrus"
)

// ReportItem represents a single item in a dynamo table for a report.
type ReportItem struct {
	ReportID   string      `json:"report_id"`
	Date       time.Time   `json:"date"`
	ReportName string      `json:"report_name"`
	Data       interface{} `json:"data"` // Arbitary structure data expressible as JSON
}

func getDynamoTable() dynamo.Table {
	// FIXME: This should be pulled from configuration
	conf := &aws.Config{Region: aws.String("us-west-2"), Endpoint: &c.Config.DynamoDBEndpoint}
	db := dynamo.New(session.New(), conf)
	table := db.Table("GoracleReportData")
	return table
}

// Save saves the ReportItem to DynamoDB
func (r *ReportItem) Save() error {
	// Before we try to save, truncate the report date
	// to align to the beginning of a UTC day
	r.Date = r.Date.UTC().Truncate(time.Hour * 24)

	logrus.Info("Save:", r)
	t := getDynamoTable()
	err := t.Put(r).Run()
	if err != nil {
		logrus.Info("Error saving item:", err)
	}
	return err
}

func deleteReport(reportID string) error {
	items, err := GetReportItems(reportID)
	if err != nil {
		return err
	}
	t := getDynamoTable()

	for _, item := range items {
		logrus.Info("Deleting:", item)
		d := item.Date
		err := t.Delete("ReportID", reportID).
			Range("Date", d).
			Run()
		if err != nil {
			logrus.Info("Error deleting report:", err)
			return err
		}
	}
	return nil
}

// GetReportItem gets a single report item for a particular day
func GetReportItem(reportID string, date time.Time) (*ReportItem, error) {
	logrus.Info(reportID, " ", date)
	t := getDynamoTable()
	var result ReportItem

	// Ensure that the date requested is pinned to the start of a UTC day,
	// to align with how data is stored in Dynamo
	date = date.UTC().Truncate(time.Hour * 24)

	err := t.Get("ReportID", reportID).
		Range("Date", dynamo.Equal, date).
		One(&result)
	logrus.Info("Result:", result)
	if err != nil {
		logrus.Info("Error getting table", err)
		return nil, err
	}
	return &result, nil
}

func GetReportItems(reportID string) ([]ReportItem, error) {
	t := getDynamoTable()
	results := []ReportItem{}
	err := t.Get("ReportID", reportID).
		All(&results)
	if err != nil {
		logrus.Info("Error getting table", err)
		return nil, err
	}

	return results, nil
}

// GetReportData returns all report items for a report in a given date range (inclusive)
func GetReportRange(reportID string, startDate time.Time, endDate time.Time) ([]ReportItem, error) {
	t := getDynamoTable()
	results := []ReportItem{}
	err := t.Get("ReportID", reportID).
		Range("Date", dynamo.Between, startDate, endDate).
		All(&results)
	if err != nil {
		logrus.Info("Error getting table", err)
		return nil, err
	}
	logrus.Info(results)
	return results, err
}
