// +build manual

package reportdata

// Note: These tests will fail without a local dynalite instance running
// AND having credentials. You will get the following error:
//
// NoCredentialProviders: no valid providers in chain. Deprecated.

import (
	"log"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"

	"code.justin.tv/availability/hms-esk/pkg/config"
)

func init() {
	// Parse in a local config for use here
	// Look for a top-level config file to use
	err := config.ParseConfig("../../config.toml")
	if err != nil {
		panic(err)
	}
}

func setupTest() {
	// Clear all reports with the test ID
	err := deleteReport("test")
	if err != nil {
		log.Fatal("Failed to clear test and set up")
	}
}

func ri() ReportItem {
	r := ReportItem{
		ReportID:   "test",
		Date:       time.Now(),
		ReportName: "Test Report information",
		Data: map[string]interface{}{
			"foo": "bar",
		},
	}
	return r
}

func TestDeleteReport(t *testing.T) {
	setupTest()

	r := ri()

	err := r.Save()
	assert.NoError(t, err, "There should be no errors saving a ReportItem")

	err = deleteReport("test")
	assert.NoError(t, err, "There should be no errors deleting a report")

	items, err := GetReportItems("test")
	assert.NoError(t, err, "There should be no errors getting report items")
	assert.Equal(t, 0, len(items), "There should be no items returned")
}

func TestSetGet(t *testing.T) {
	r := ri()

	err := r.Save()
	assert.NoError(t, err, "There should be no errors saving a ReportItem")

	result, err := GetReportItem("test", r.Date)
	assert.NoError(t, err, "There should be no errors getting a ReportItem")
	assert.Equal(t, r, *result, "The ReportItem retrieved should be the same as the one saved")
}

func TestGetReportItems(t *testing.T) {
	setupTest()
	// pick a starting time that isnt truncated
	initialTime := time.Date(2009, time.November, 10, 17, 5, 2, 3, time.UTC)
	r1 := ReportItem{
		ReportID:   "test",
		Date:       initialTime,
		ReportName: "Test Report information, day 1",
		Data: map[string]interface{}{
			"foo": "bar",
		},
	}
	r2 := ReportItem{
		ReportID:   "test",
		Date:       initialTime.Add(27 * time.Hour),
		ReportName: "Test Report information, day 2",
		Data: map[string]interface{}{
			"foo": "bar",
		},
	}
	r3 := ReportItem{
		ReportID:   "test",
		Date:       initialTime.Add(50 * time.Hour),
		ReportName: "Test Report information, day 3",
		Data: map[string]interface{}{
			"foo": "bar",
		},
	}
	err := r1.Save()
	err = r2.Save()
	err = r3.Save()
	assert.NoError(t, err, "There shouldn't be errors saving the items initially")

	// Just get all the reports
	ris, err := GetReportItems("test")
	assert.NoError(t, err, "There should be no errors fetching all test reports")
	assert.Equal(t, 3, len(ris))

	// Fetch only the 2 more recent reports (remember the timestamp gets truncated down,
	// so initialTime as a start date is in front of the first report
	ris, err = GetReportRange("test", initialTime, initialTime.Add(100*time.Hour))
	assert.NoError(t, err, "There should be no errors fetching a range of test reports")
	assert.Equal(t, 2, len(ris))
	assert.Equal(t, ris[0].Date.Hour(), 0)
	assert.Equal(t, ris[0].Date.Minute(), 0)
	assert.Equal(t, ris[0].Date.Second(), 0)
	assert.Equal(t, ris[0].Date.Nanosecond(), 0)
	assert.Equal(t, ris[0].Date.Day(), 11)
	assert.Equal(t, ris[1].Date.Hour(), 0)
	assert.Equal(t, ris[1].Date.Minute(), 0)
	assert.Equal(t, ris[1].Date.Second(), 0)
	assert.Equal(t, ris[1].Date.Nanosecond(), 0)
	assert.Equal(t, ris[1].Date.Day(), 12)

}
