package handler

import (
	"io/ioutil"
	"net/http"

	"github.com/prometheus/prometheus/prompb"
	log "github.com/sirupsen/logrus"

	"code.justin.tv/availability/necrophos/config"
	"code.justin.tv/availability/necrophos/metrics"
)

type PrometheusProxy struct {
	Config *config.Config
}

func NewPrometheusProxy(c *config.Config) *PrometheusProxy {
	return &PrometheusProxy{
		Config: c,
	}
}

func (pp *PrometheusProxy) HandleReadRequest(w http.ResponseWriter, r *http.Request) {
	// Read the compressed request body
	compressed, err := ioutil.ReadAll(r.Body)
	if err != nil {
		log.Errorf("error reading request: %s", err.Error())
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}

	// convert request body into prompb.ReadRequest
	req, err := metrics.DecodePrometheusReadRequest(compressed)
	if err != nil {
		log.Infof("bad request received: %s", err.Error())
		http.Error(w, err.Error(), http.StatusBadRequest)
		return
	}

	// The ReadRequest contains an array of Query structs, which have start time,
	// end time, and a set of label matchers
	queries := req.GetQueries()

	log.Infof("received %d queries to process", len(queries))

	// For each query, create a QueryHandler, which will create a plan for how to fetch the results
	queryHandlers := make([]*QueryHandler, 0, len(queries))
	for _, q := range queries {
		qh := NewQueryHandler(q)
		// Make a plan for the query (where to backend to, what retention, etc)
		queryHandlers = append(queryHandlers, qh)
	}

	// For each QueryHandler, Execute() the plan
	for _, qh := range queryHandlers {
		go qh.Execute()
	}

	// Read the results of the QueryHandlers back in the same order
	// the queries were issued (which was the same order they were received)
	queryResults := make([]*prompb.QueryResult, 0)
	for _, qh := range queryHandlers {
		queryResult, err := qh.GetQueryResults()
		if err != nil {
			log.Errorf("error querying cloudwatch: %s", err.Error())
			http.Error(w, err.Error(), http.StatusInternalServerError)
			return
		}
		queryResults = append(queryResults, queryResult)
	}
	log.Infof("query results successfully fetched; got %d QueryResult objects", len(queryResults))

	// Protobuf encode + snappy compress the results, and return!
	// first make the datastructure
	finalReadResp := &prompb.ReadResponse{Results: queryResults}

	// now protobuf it
	compressedData, err := metrics.EncodePrometheusReadResponse(finalReadResp)
	if err != nil {
		log.Errorf("error encoding read response: %s", err.Error())
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}
	log.Infof("length of response: %d", len(compressedData))
	w.Header().Set("Content-Type", "application/x-protobuf")
	w.Header().Set("Content-Encoding", "snappy")
	_, err = w.Write(compressedData)
	if err != nil {
		log.Errorf("error writing response: %s", err.Error())
		http.Error(w, err.Error(), http.StatusInternalServerError)
	}
}

func (pp *PrometheusProxy) HandleWriteRequest(w http.ResponseWriter, r *http.Request) {
	// decode
	// convert into PutMetricDataInput
	// submit the blob of datapoints
	compressed, err := ioutil.ReadAll(r.Body)
	if err != nil {
		log.Errorf("error receiving write request: %s", err.Error())
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}

	req, err := metrics.DecodePrometheusWriteRequest(compressed)
	if err != nil {
		log.Infof("error decoding write request: %s", err.Error())
		http.Error(w, err.Error(), http.StatusBadRequest)
		return
	}

	err = metrics.PutCloudwatchData(req.Timeseries)
	if err != nil {
		log.Errorf("error writing data to cloudwatch: %s", err.Error())
		http.Error(w, err.Error(), http.StatusInternalServerError)
	}
}
