#!/bin/bash

set -e -x

PROMTEAM=$(sudo facter -p prometheus_team)
PROMTEAMDIR="/opt/twitch/prometheus-config/current/teams/${PROMTEAM}"
PROMLIVECONFIG="/etc/prometheus/prometheus.yaml"
PROMNEWCONFIG="/opt/twitch/prometheus-config/current/prometheus.new.yaml"
PROMBASECONFIG="/etc/prometheus/prometheus.base.yaml"
PROMRULESCONFIG="/etc/prometheus/rules.d/rules.yaml"
PROMNEWRULESCONFIG="/opt/twitch/prometheus-config/current/rules.new.yaml"

# Check that the team exists
[[ ! -d $PROMTEAMDIR ]] && echo "ERROR: unknown team '$PROMTEAM'" && exit 1

# Checks for a prometheus scrape config
[[ ! -f "${PROMTEAMDIR}/scrapes.yaml" ]] && echo "ERROR: no scrapes.yaml config found for team '$PROMTEAM'" && exit 1

# Checks for a base config
[[ ! -f $PROMBASECONFIG ]] && echo "ERROR: no base configuration found on host for team '$PROMTEAM'" && exit 1

# Create the prometheus.yaml candidate
/opt/twitch/prometheus-config/current/generate.py --team $PROMTEAM --config-base /etc/prometheus/prometheus.base.yaml --output $PROMNEWCONFIG --rules-output $PROMNEWRULESCONFIG
[[ "$?" != "0" ]] && echo "ERROR: failure generating prometheus configuration for team '$PROMTEAM'" && exit 1

# Verify that the prometheus config file looks good
/usr/bin/promtool check config $PROMNEWCONFIG
[[ "$?" != "0" ]] && echo "ERROR: generated prometheus configuration does not pass promtool syntax check for team '$PROMTEAM'" && exit 1


[[ ! -f $PROMRULESCONFIG ]] || ! diff $PROMNEWRULESCONFIG $PROMRULESCONFIG
if [[ "$?" == "0" ]]; then
    echo "INFO: writing new rules file for team '$PROMTEAM', host='$(hostname -f)'"
    cat $PROMNEWRULESCONFIG | sudo tee $PROMRULESCONFIG
    SHOULD_RELOAD="yes"
fi

# If the diff has changes, swap the file and reload
if ! $(diff $PROMNEWCONFIG $PROMLIVECONFIG) ; then
  # Just overwrite the file contents so we don't need to muck with file
  # permissions and ownership
  echo "INFO: writing new prometheus configuration for team '$PROMTEAM', host='$(hostname -f)'"
  cat $PROMNEWCONFIG | sudo tee $PROMLIVECONFIG
  SHOULD_RELOAD="yes"
fi

if [[ -n "$SHOULD_RELOAD" ]] ; then 
  # HUP the prometheus server to reload the config
  sudo kill -HUP $(pgrep prometheus)
fi

# Cleanup
rm -rf $PROMNEWCONFIG
rm -rf $PROMNEWRULESCONFIG


# It's possible that we get here without actually doing anything,
# and thats OK because there are lots of different prometheus servers,
# so a config change for one team shouldn't trigger global reloads :)

