package main

import (
	"os"
	"strings"
	"unicode"

	"code.justin.tv/systems/plucker/digestor"
	"github.com/BurntSushi/toml"
	"github.com/pkg/errors"
)

const pluckerEventRulesHeader = `# This file was dynamically generated by
# https://git-aws.internal.justin.tv/availability/site-performance-metrics`

const (
	benchmarkCompleteTransition = "benchmark_complete_transition"
	networkRequest              = "network_request"
)

func pluckerEventRules(pages []Page, apiRequests []APIRequest, cdnRequests []CDNRequest,
	dnsRequests []DNSRequest, countries []Country) digestor.EventRules {

	eventRules := make(digestor.EventRules)
	addPageEventRules(eventRules, pages, countries)
	addAPIRequestEventRules(eventRules, apiRequests, countries)
	addCDNRequestEventRules(eventRules, cdnRequests, countries)
	addDNSRequestEventRules(eventRules, dnsRequests, countries)
	return eventRules
}

func writeEventRules(eventRules digestor.EventRules, filename string) error {
	f, err := os.Create(filename)
	if err != nil {
		return errors.Wrapf(err, "failed to create file %q", filename)
	}
	defer f.Close()

	_, err = f.WriteString(pluckerEventRulesHeader + "\n\n")
	if err != nil {
		return errors.Wrapf(err, "failed to write header to %q", filename)
	}

	err = toml.NewEncoder(f).Encode(eventRules)
	return errors.Wrapf(err, "failed to encode Plucker event rules to %q", filename)
}

func graphiteSafeString(str string) string {
	safeStr := strings.Map(func(r rune) rune {
		if unicode.IsLetter(r) || unicode.IsDigit(r) {
			return r
		}
		return '_'
	}, str)

	for strings.Contains(safeStr, "__") {
		safeStr = strings.Replace(safeStr, "__", "_", -1)
	}
	return safeStr
}
