package main

import (
	"encoding/csv"
	"flag"
	"log"
	"net/url"
	"os"

	"github.com/pkg/errors"
)

var pagesCSV string
var apiRequestsCSV string
var cdnRequestsCSV string
var dnsRequestsCSV string
var countriesCSV string
var destination string

func init() {
	flag.StringVar(&pagesCSV, "pages", "pages.csv", "provied list of pages to monitor")
	flag.StringVar(&apiRequestsCSV, "api", "api.csv", "provied list of api requests to monitor")
	flag.StringVar(&cdnRequestsCSV, "cdn", "cdn.csv", "provied list of cdn requests to monitor")
	flag.StringVar(&dnsRequestsCSV, "dns", "dns.csv", "provied list of dns requests to monitor")
	flag.StringVar(&countriesCSV, "countries", "countries.csv", "provied list of countries to monitor")
	flag.StringVar(&destination, "destination", "site_performance.toml", "destination of generated Plucker event rules")
	flag.Parse()
}

func main() {
	pages, err := readPages(pagesCSV)
	if err != nil {
		log.Fatalf("Unexpected error while reading pages: %v", err)
	}

	apiRequests, err := readAPIRequests(apiRequestsCSV)
	if err != nil {
		log.Fatalf("Unexpected error while reading api requests: %v", err)
	}

	cdnRequests, err := readCDNRequests(cdnRequestsCSV)
	if err != nil {
		log.Fatalf("Unexpected error while reading cdn requests: %v", err)
	}

	dnsRequests, err := readDNSRequests(dnsRequestsCSV)
	if err != nil {
		log.Fatalf("Unexpected error while reading dns requests: %v", err)
	}

	countries, err := readCountries(countriesCSV)
	if err != nil {
		log.Fatalf("Unexpected error while reading countries: %v", err)
	}

	eventRules := pluckerEventRules(pages, apiRequests, cdnRequests, dnsRequests, countries)
	err = writeEventRules(eventRules, destination)
	if err != nil {
		log.Fatalf("Unexpected error while writing event rules: %v", err)
	}
}

func readPages(filename string) ([]Page, error) {
	lines, err := readCSV(filename)
	if err != nil {
		return nil, errors.Wrap(err, "failed to read CSV")
	}

	pages := make([]Page, 0, len(lines))
	for _, line := range lines[1:] {
		page := Page{Hostname: "www.twitch.tv", Destination: line[0]}
		pages = append(pages, page)
	}
	return pages, nil
}

func readAPIRequests(filename string) ([]APIRequest, error) {
	lines, err := readCSV(filename)
	if err != nil {
		return nil, errors.Wrap(err, "failed to read CSV")
	}

	apiRequests := make([]APIRequest, 0, len(lines))
	for _, line := range lines[1:] {
		u, err := url.Parse(line[0])
		if err != nil {
			return nil, errors.Wrapf(err, "failed to parse url %q", line[0])
		}

		apiRequest := APIRequest{Hostname: u.Hostname(), Path: u.Path}
		apiRequests = append(apiRequests, apiRequest)
	}
	return apiRequests, nil
}

func readCDNRequests(filename string) ([]CDNRequest, error) {
	lines, err := readCSV(filename)
	if err != nil {
		return nil, errors.Wrap(err, "failed to read CSV")
	}

	cdnRequests := make([]CDNRequest, 0, len(lines))
	for _, line := range lines[1:] {
		u, err := url.Parse(line[0])
		if err != nil {
			return nil, errors.Wrapf(err, "failed to parse url %q", line[0])
		}

		cdnRequest := CDNRequest{Hostname: u.Hostname(), Path: u.Path}
		cdnRequests = append(cdnRequests, cdnRequest)
	}
	return cdnRequests, nil
}

func readDNSRequests(filename string) ([]DNSRequest, error) {
	lines, err := readCSV(filename)
	if err != nil {
		return nil, errors.Wrap(err, "failed to read CSV")
	}

	dnsRequests := make([]DNSRequest, 0, len(lines))
	for _, line := range lines[1:] {
		dnsRequest := DNSRequest{Hostname: line[0]}
		dnsRequests = append(dnsRequests, dnsRequest)
	}
	return dnsRequests, nil
}

func readCountries(filename string) ([]Country, error) {
	lines, err := readCSV(filename)
	if err != nil {
		return nil, errors.Wrap(err, "failed to read CSV")
	}

	countries := make([]Country, 0, len(lines))
	for _, line := range lines[1:] {
		country := Country{Code: line[0]}
		countries = append(countries, country)
	}
	return countries, nil
}

func readCSV(filename string) ([][]string, error) {
	f, err := os.Open(filename)
	if err != nil {
		return nil, errors.Wrapf(err, "failed to read file %q", filename)
	}
	defer f.Close()

	lines, err := csv.NewReader(f).ReadAll()
	return lines, errors.Wrapf(err, "failed to parse %q as CSV", filename)
}
