package digestor

import (
	"bytes"
	"fmt"
	"hash/fnv"
	"strconv"
	"strings"
	"text/template"
	"unicode"

	log "github.com/Sirupsen/logrus"

	"code.justin.tv/systems/plucker/kinesis"
	"github.com/cactus/go-statsd-client/statsd"
)

// reference time Mon Jan 2 15:04:05 MST 2006
const dateParseFormat = "2006-01-02 15:04:05"

var (
	// TIMER statsd type
	TIMER = "timer"
	// GAUGE statsd type
	GAUGE = "gauge"
	// COUNTER statsd type
	COUNTER = "counter"
	// SET statsd type
	SET = "set"
	// INT Spade integer
	INT = "bigint"
	// FLOAT Spade float
	FLOAT = "float"
)

// StatsdDigestor state object
type StatsdDigestor struct {
	Config        *Config
	StatsEndpoint statsd.Statter
}

// Stat represents a metric value of some event pulled from spade
type Stat struct {
	Value     int64
	EventName string // as named in Spade
	StatName  string // optionally configured alternative name
}

// NewStatsdDigestor returns a new plucker object with loaded schemas
func NewStatsdDigestor(config *Config) (*StatsdDigestor, error) {

	sd := &StatsdDigestor{
		Config:        config,
		StatsEndpoint: nil, // Set this in a separate call
	}

	err := sd.verifyConfig()
	if err != nil {
		return nil, fmt.Errorf("Configuration validation error: %s", err.Error())
	}

	return sd, nil
}

// WithStatter adds a statsd endpoint to a statsd digestor
func (sd *StatsdDigestor) WithStatter(s statsd.Statter) *StatsdDigestor {
	sd.StatsEndpoint = s
	return sd
}

// Handle implements kinesis message handler
func (sd *StatsdDigestor) Handle(event *kinesis.Event) error {

	if !sd.isWhiteListed(event) {
		log.Infof("Received Kinesis record for event not in configuration: %q\n", event.Name)
		return nil
	}

	// Commented out because this would make too much noise
	//log.Infof("[Event]: Processing Schema %s", event.SchemaName())

	// Build all of the stats that this event should produce
	stats, err := sd.buildStats(event)
	if err != nil {
		return fmt.Errorf("Error building stats: %s", err.Error())
	}
	for _, stat := range stats {
		log.Debug(stat)
	}
	err = sd.Flush(stats)
	if err != nil {
		return fmt.Errorf("Error sending stats: %s", err.Error())
	}
	return nil
}

// Flush dumps stats to the underlying statsd instance
func (sd *StatsdDigestor) Flush(stats []*Stat) error {
	var err error
	for index, stat := range stats {
		statType := sd.Config.EventRules[stat.EventName][index].StatType
		switch statType {
		case GAUGE:
			err = sd.StatsEndpoint.Gauge(stat.StatName, stat.Value, 1)
		case SET:
			err = sd.StatsEndpoint.SetInt(stat.StatName, stat.Value, 1)
		case COUNTER:
			err = sd.StatsEndpoint.Inc(stat.StatName, stat.Value, 1)
		case TIMER:
			err = sd.StatsEndpoint.Timing(stat.StatName, stat.Value, 1)
		default:
			return fmt.Errorf("Error flushing stat due to bad type: %s", statType)
		}
		if err != nil {
			return fmt.Errorf("Error pushing to Statter: %s", err.Error())
		}
	}
	return nil
}

func (sd *StatsdDigestor) buildStats(event *kinesis.Event) ([]*Stat, error) {
	eventConfigs := sd.Config.EventRules[event.Name]

	stats := make([]*Stat, 0, len(eventConfigs))

	for _, eventConfig := range eventConfigs {
		if sd.eventMatchesEqualFilter(event, eventConfig) && sd.eventMatchesRegexFilter(event, eventConfig) {
			stat, err := sd.buildStat(event, eventConfig)
			if err == nil {
				stats = append(stats, stat)
			} else {
				log.Warnf("Error building a stat: %s", err.Error())
			}
		}
	}
	return stats, nil
}

func (sd *StatsdDigestor) eventMatchesEqualFilter(event *kinesis.Event, eventConfig *EventConfig) bool {
	for filterProperty, validValues := range eventConfig.FilterConfig.Equal {
		value, found := event.Fields[filterProperty]
		if !found {
			log.Warnf("Event %q read from Kinesis has no column %q", event.Name, filterProperty)
			return false
		}

		if !valueMatchesExactly(value, validValues) {
			return false
		}
	}
	return true
}

func (sd *StatsdDigestor) eventMatchesRegexFilter(event *kinesis.Event, eventConfig *EventConfig) bool {
	for filterProperty, validPatterns := range eventConfig.FilterConfig.Regex {
		value, found := event.Fields[filterProperty]
		if !found {
			log.Warnf("Event %q read from Kinesis has no column %q", event.Name, filterProperty)
			return false
		}

		if !valueMatchesPattern(value, validPatterns) {
			return false
		}
	}
	return true
}

// BuildStats receives a CSV line and turns it into a Stat
func (sd *StatsdDigestor) buildStat(event *kinesis.Event, eventConfig *EventConfig) (*Stat, error) {
	statName, err := sd.statName(event, eventConfig)
	if err != nil {
		return nil, fmt.Errorf("Error determining stat name: %s", err.Error())
	}

	var statValue int64
	switch eventConfig.StatType {
	case "gauge":
		statValue, err = sd.parseGaugeStatValue(event, eventConfig)
	case "set":
		statValue, err = sd.parseSetStatValue(event, eventConfig)
	case "counter":
		statValue, err = sd.parseCounterStatValue(event, eventConfig)
	case "timer":
		statValue, err = sd.parseTimerStatValue(event, eventConfig)
	default:
		err = fmt.Errorf("StatType not recognized for event: %s", event.Name)
	}

	if err != nil {
		return nil, fmt.Errorf("Error parsing stat value: %s", err.Error())
	}

	newStat := &Stat{
		Value:     statValue,
		EventName: string(event.Name),
		StatName:  statName,
	}
	return newStat, nil
}

// StatName determines the correct stat name for the event. If no alias is
// provided, the stat name will simply be the event name. Otherwise, the alias
// is treated as a template and parsed.
func (sd *StatsdDigestor) statName(event *kinesis.Event, eventConfig *EventConfig) (string, error) {
	if eventConfig.Alias == "" {
		return string(event.Name), nil
	}

	t, err := template.New("").Parse(eventConfig.Alias)
	if err != nil {
		return "", fmt.Errorf("Error parsing template %q: %s", eventConfig.Alias, err)
	}

	// We require that any data provided when evaluating a template also
	// appear in the equality filter. This is to ensure that the possible
	// keyspace generated in Graphite is bounded.
	templateData := make(map[string]string)
	for filterProperty, validValues := range eventConfig.FilterConfig.Equal {
		value, found := event.Fields[filterProperty]
		if found && valueMatchesExactly(value, validValues) {
			templateData[filterProperty] = value
		}
	}

	var b bytes.Buffer
	err = t.Execute(&b, templateData)
	if err != nil {
		return "", fmt.Errorf("Error executing template %q: %s", eventConfig.Alias, err)
	}

	name := b.String()
	if strings.IndexFunc(name, unicode.IsSpace) != -1 {
		return "", fmt.Errorf("Stat name %q contains a space", name)
	}
	return name, nil
}

func (sd *StatsdDigestor) parseGaugeStatValue(event *kinesis.Event, eventConfig *EventConfig) (int64, error) {
	var statValue int64
	var err error
	var statValueString, found = event.Fields[eventConfig.ValueField]
	if !found {
		return 0, fmt.Errorf("value_field %q not found for event %q", eventConfig.ValueField, event.Name)
	}

	statValue, err = strconv.ParseInt(statValueString, 10, 64)
	if err != nil {
		return 0, fmt.Errorf("error parsing stat value for gauge event %q with value field %q: %q", event.Name, eventConfig.ValueField, err.Error())
	}
	return statValue, nil
}

func (sd *StatsdDigestor) parseSetStatValue(event *kinesis.Event, eventConfig *EventConfig) (int64, error) {
	var err error
	var statValueString, found = event.Fields[eventConfig.ValueField]
	if !found {
		return 0, fmt.Errorf("value_field %q not found for event %q", eventConfig.ValueField, event.Name)
	}

	// if using sets, create a unique ID number for the field by hashing
	// we hash instead of passing the string directly because the Stat struct
	// expects an integer for the Value field
	hasher := fnv.New64a()
	_, err = hasher.Write([]byte(statValueString))
	if err != nil {
		return 0, fmt.Errorf("Error hashing string for set event: %q: %q", event.Name, err.Error())
	}

	return int64(hasher.Sum64()), nil
}

func (sd *StatsdDigestor) parseCounterStatValue(event *kinesis.Event, eventConfig *EventConfig) (int64, error) {
	var statValue float64
	var err error

	// The idea here is that for something like `pageviews`, we aren't parsing
	// any actual numbers from the event itself... we just want to count that event as 1,
	// which by itself represents a single pageview.
	// Thus, we don't specify a column that holds data for pageviews (ValueField),
	// and so plucker implicitly assumes that we want to just count the number of
	// rows, i.e. we set statValue to 1.
	if eventConfig.ValueField == "" {
		return 1, nil
	}

	// Otherwise, parse out the ValueField
	var statValueString, found = event.Fields[eventConfig.ValueField]
	if !found {
		return 0, fmt.Errorf("value_field %q not found for event %q", eventConfig.ValueField, event.Name)
	}

	// Parse a float just so we dont get parse errors in the event that the field is actually a float type
	statValue, err = strconv.ParseFloat(statValueString, 64)
	if err != nil {
		return 0, fmt.Errorf("Error parsing stat value for counter event %q with value field %q: %q", event.Name, eventConfig.ValueField, err.Error())
	}

	return int64(statValue), nil
}

func (sd *StatsdDigestor) parseTimerStatValue(event *kinesis.Event, eventConfig *EventConfig) (int64, error) {
	var statValue float64
	var floatStatValue float64
	var err error
	var statValueString, found = event.Fields[eventConfig.ValueField]
	if !found {
		return 0, fmt.Errorf("value_field %q not found for event %q", eventConfig.ValueField, event.Name)
	}

	// Keep everything as floats until just before returning the value
	var statValueMultiplier float64 = 1
	if eventConfig.Multiplier != 0 {
		statValueMultiplier = float64(eventConfig.Multiplier)
	}

	// Just always parse as a float because why not
	floatStatValue, err = strconv.ParseFloat(statValueString, 64)
	if err == nil {
		statValue = float64(statValueMultiplier) * floatStatValue
	}
	if err != nil {
		return 0, fmt.Errorf("Error parsing stat value for timer event %q with value field %q: %q", event.Name, eventConfig.ValueField, err.Error())
	}

	// Cast to int64 just before returning
	return int64(statValue), nil
}

// IsWhiteListed checks to see if the event matches any of those we are interested
// in based on configuration
func (sd *StatsdDigestor) isWhiteListed(event *kinesis.Event) bool {
	_, ok := sd.Config.EventRules[event.Name]
	return ok
}

// ensures that the collector configuration is valid for all information we have
func (sd *StatsdDigestor) verifyConfig() error {
	if sd.Config == nil {
		return fmt.Errorf("Config cannot be empty")
	}

	// Some general rules:
	// - If an event is a gauge, it must have a ValueField set
	// - The ValueField must exist if set
	for eventName, eventRules := range sd.Config.EventRules {
		for _, eventRule := range eventRules {
			// gauges require a value field
			if eventRule.ValueField == "" && eventRule.StatType == "gauge" {
				return fmt.Errorf("Event %q is listed as a gauge without a value field", eventName)
			}
			// sets require a value field
			if eventRule.ValueField == "" && eventRule.StatType == "set" {
				return fmt.Errorf("Event %q is listed as a set, but has no value field", eventName)
			}
			// timers require a value field
			if eventRule.ValueField == "" && eventRule.StatType == "timer" {
				return fmt.Errorf("Event %q is listed as a timer, but has no value field", eventName)
			}
		}
	}
	return nil
}
