package digestor

import (
	"fmt"
	"testing"

	"code.justin.tv/systems/plucker/kinesis"

	. "github.com/smartystreets/goconvey/convey"
)

func TestBuildStats(t *testing.T) {

	Convey("When building stats for an event", t, func() {
		config := DefaultConfig()
		eventConfig := &EventConfig{
			StatType:   "timer",
			ValueField: "duration",
			Alias:      "graphite.key",
		}
		config.EventRules["event_name"] = []*EventConfig{eventConfig}

		event := &kinesis.Event{
			Name: "event_name",
			Fields: map[string]string{
				"request_url": "https://api.twitch.tv/api/me",
				"duration":    "1000",
				"country":     "DE",
			},
		}

		Convey("When creating the stat name", func() {

			Convey("When an alias is not provided", func() {
				eventConfig.Alias = ""

				digestor, err := NewStatsdDigestor(config)
				So(err, ShouldBeNil)

				statName, err := digestor.statName(event, eventConfig)
				So(err, ShouldBeNil)
				So(statName, ShouldEqual, event.Name)
			})

			Convey("When a simple alias is provided", func() {
				eventConfig.Alias = "simple.alias"

				digestor, err := NewStatsdDigestor(config)
				So(err, ShouldBeNil)

				statName, err := digestor.statName(event, eventConfig)
				So(err, ShouldBeNil)
				So(statName, ShouldEqual, eventConfig.Alias)
			})

			Convey("When a templatized alias is provided", func() {

				Convey("When a requested field is not in the filter", func() {
					eventConfig.Alias = "templatized.{{.country}}.alias"

					digestor, err := NewStatsdDigestor(config)
					So(err, ShouldBeNil)

					_, err = digestor.statName(event, eventConfig)
					So(err, ShouldNotBeNil)
				})

				Convey("When all requsted fields are in filters", func() {
					eventConfig.Alias = "templatized.{{.country}}.alias"
					eventConfig.FilterConfig.Equal = map[string][]string{
						"country": []string{"US", "RU", "DE", "BR", "TW"},
					}

					digestor, err := NewStatsdDigestor(config)
					So(err, ShouldBeNil)

					statName, err := digestor.statName(event, eventConfig)
					So(err, ShouldBeNil)
					So(statName, ShouldEqual, fmt.Sprintf("templatized.%s.alias", event.Fields["country"]))
				})
			})
		})

		Convey("When filtering", func() {

			Convey("When an equality filter is provided", func() {
				config.EventRules["event_name"][0].FilterConfig.Equal = map[string][]string{
					"country": []string{"US", "DE", "RU"},
				}

				digestor, err := NewStatsdDigestor(config)
				So(err, ShouldBeNil)

				Convey("When there is a match", func() {
					stats, err := digestor.buildStats(event)
					So(err, ShouldBeNil)
					So(len(stats), ShouldEqual, 1)
					So(stats[0].StatName, ShouldEqual, eventConfig.Alias)
				})

				Convey("When there is not a match", func() {
					event.Fields["country"] = "BR"

					stats, err := digestor.buildStats(event)
					So(err, ShouldBeNil)
					So(len(stats), ShouldEqual, 0)
				})
			})

			Convey("When a regex filter is provided", func() {
				config.EventRules["event_name"][0].FilterConfig.Regex = map[string][]string{
					"request_url": []string{"^https://api.twitch.tv/viewer/info.json(\\?|$)", "^https://api.twitch.tv/api/me(\\?|$)"},
				}

				digestor, err := NewStatsdDigestor(config)
				So(err, ShouldBeNil)

				Convey("When there is a match", func() {
					stats, err := digestor.buildStats(event)
					So(err, ShouldBeNil)
					So(len(stats), ShouldEqual, 1)
					So(stats[0].StatName, ShouldEqual, eventConfig.Alias)
				})

				Convey("When there is not a match", func() {
					event.Fields["request_url"] = "https://api.twitch.tv/not/a/match"

					stats, err := digestor.buildStats(event)
					So(err, ShouldBeNil)
					So(len(stats), ShouldEqual, 0)
				})
			})

			Convey("When all filter types are provided", func() {
				config.EventRules["event_name"][0].FilterConfig.Equal = map[string][]string{
					"country": []string{"US", "DE", "RU"},
				}

				config.EventRules["event_name"][0].FilterConfig.Regex = map[string][]string{
					"request_url": []string{"^https://api.twitch.tv/api/me(\\?|$)", "^https://api.twitch.tv/kraken/me(\\?|$)"},
				}

				digestor, err := NewStatsdDigestor(config)
				So(err, ShouldBeNil)

				Convey("When there is a match", func() {
					stats, err := digestor.buildStats(event)
					So(err, ShouldBeNil)
					So(len(stats), ShouldEqual, 1)
					So(stats[0].StatName, ShouldEqual, eventConfig.Alias)
				})

				Convey("When there is not an equality match", func() {
					event.Fields["country"] = "BR"

					stats, err := digestor.buildStats(event)
					So(err, ShouldBeNil)
					So(len(stats), ShouldEqual, 0)
				})

				Convey("When there is not a regex match", func() {
					event.Fields["request_url"] = "https://api.twitch.tv/not/a/match"

					stats, err := digestor.buildStats(event)
					So(err, ShouldBeNil)
					So(len(stats), ShouldEqual, 0)
				})
			})
		})
	})
}
