package kinesis

import (
	"bytes"
	"compress/flate"
	"encoding/json"
	"fmt"
	"io"
)

// Record is the JSON object returned from Kinsumer.Next().
type Record struct {
	UUID    string
	Version int
	Data    []byte
}

// NewRecordFromBytes converts a raw kinesis record
// into a record struct via JSON unmarshaling
func NewRecordFromBytes(b []byte) (*Record, error) {
	var record Record
	// Unmarshal raw data into Record struct
	err := json.Unmarshal(b, &record)
	if err != nil {
		return nil, err
	}
	return &record, nil
}

func (r *Record) validateFormat() error {
	// Perform simple validation on the data, since we sort of
	// hard code handling for the only existing version/schema of these records
	if r.Version != 1 {
		return fmt.Errorf("kinesis event version is not 1, instead: %d", r.Version)
	}
	if r.Data[0] != 1 {
		return fmt.Errorf("data compression bit was not 1, instead: %d", r.Data[0])
	}
	return nil
}

func (r *Record) decompress(b *bytes.Buffer) error {
	deflator := flate.NewReader(bytes.NewBuffer(r.Data[1:]))
	_, err := io.Copy(b, deflator)
	// TODO: Determine if this is okay to leave commented out
	// Rationale: Since this is a short function call and the
	// scope of `deflator` is short-lived, the GC should clean
	// up `deflator`; no need to invoke a Close().
	// err = deflator.Close()
	// if err != nil {
	// 	log.Errorf("Error closing deflator: %s", err.Error())
	// }
	// This will be nil if there is not an error, else
	// it will hold the error
	return err
}
