package kinesis

import (
	"bytes"
	"fmt"
	"os"

	log "github.com/Sirupsen/logrus"

	"github.com/TwitchScience/kinsumer"
	"github.com/TwitchScience/kinsumer/statsd"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/kinesis"
	"github.com/twinj/uuid"
)

// Worker contains a Kinsumer and a callback to handle events
type Worker struct {
	Kinsumer *kinsumer.Kinsumer
	Handle   func(*Event) error
	records  chan []byte
}

// NewWorker returns a new worker
func NewWorker(config *Config, handler func(*Event) error) (*Worker, error) {
	var (
		stats kinsumer.StatReceiver = &kinsumer.NoopStatReceiver{}
		err   error
	)
	if len(config.StreamName) == 0 {
		log.Fatalln("stream name parameter is required")
	}

	hostname, err := os.Hostname()
	if err != nil {
		log.Fatalln("error retrieving hostname")
	}

	statsdPrefix := fmt.Sprintf("plucker.%s.%s", config.Environment, hostname)

	if len(config.StatsdEndpoint) > 0 {
		stats, err = statsd.New(config.StatsdEndpoint, statsdPrefix)
		if err != nil {
			log.Fatalf("Error creating statsd object: %v", err)
		}
	}

	cfg := aws.NewConfig().WithRegion("us-west-2")

	mainSession := session.New(cfg)

	kinsumerConfig := kinsumer.NewConfig().WithStats(stats)

	kin := kinesis.New(mainSession)
	dyn := dynamodb.New(mainSession)

	// kinsumer needs a way to differentiate between running clients, generally you want to use information
	// about the machine it is running on like ip. For this example we'll use a uuid
	name := uuid.NewV4().String()

	k, err := kinsumer.NewWithInterfaces(kin, dyn, config.StreamName, config.StreamName, name, kinsumerConfig)
	if err != nil {
		log.Fatalf("Error creating kinsumer: %v", err)
	}

	worker := &Worker{
		Kinsumer: k,
		Handle:   handler,
		records:  make(chan []byte, 1000),
	}

	return worker, nil
}

// Fetch reads from a Kinesis stream
// performs minimal validation after fetching the record
// Difficult to write tests for
func (w *Worker) Fetch() {
	log.Debug("starting kinesis fetcher")
	err := w.Kinsumer.Run()
	if err != nil {
		log.Fatalf("Error calling Kinsumer.Run(): %v", err)
	}

	// Infinite fetch loop
	for {
		// Fetch raw bytes from the kinesis stream
		log.Debug("fetching data from stream")
		recordData, err := w.Kinsumer.Next()
		log.Debugf("got data from stream: %v", recordData)
		if err != nil {
			log.Fatalf("error reading from Kinesis stream: %v", err)
		}
		w.records <- recordData
		if recordData == nil {
			log.Debug("exitting kinesis Fetch() loop")
			return
		}
	}
}

// Process parses and ships stats from Kinesis records
func (w *Worker) Process() {
	for {
		// Read raw record data from Run()
		recordData := <-w.records

		// Treat a nil going through the pipe as a signal to quit
		if recordData == nil {
			log.Debug("exitting kinesis Process() loop")
			return
		}

		// Generate struct from kinesis data
		record, err := NewRecordFromBytes(recordData)
		if err != nil {
			log.Errorf("error unmarshaling raw kinesis data: %s", err.Error())
			continue
		}

		// Ensure the record is formatted properly
		err = record.validateFormat()
		if err != nil {
			log.Warnf("Invalid kinesis record format: %s", err.Error())
			continue
		}

		// Decompress payload data
		var payload bytes.Buffer
		err = record.decompress(&payload)
		if err != nil {
			log.Errorf("Error decompressing record payload: %s", err.Error())
			continue
		}

		// Unmarshal record data segment into Event structs
		events, err := NewEventsFromPayload(payload.Bytes())
		if err != nil {
			log.Errorf("Error unmarshaling record payload into spade events: %s", err.Error())
			continue
		}

		// Process each event individually
		for _, event := range events {
			// Calls the given handler function for the worker
			err := w.Handle(event)
			if err != nil {
				// Don't have a `continue` in this err block since
				// we can still try to process the rest of the events
				// in this loop
				log.Errorf("error handling Kinesis event '%v': %v", event, err)
			}
		}
	}
}

// Stop will shutdown the kinesis fetch and process pipeline
func (w *Worker) Stop() {
	log.Info("received stop signal, shutting down.")
	w.Kinsumer.Stop()
}
