package main

import (
	"bytes"
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"os/signal"
	"path/filepath"
	"sync"
	"syscall"

	log "github.com/Sirupsen/logrus"

	"github.com/BurntSushi/toml"
	"github.com/imdario/mergo"

	"code.justin.tv/systems/plucker/digestor"
	"code.justin.tv/systems/plucker/kinesis"
)

// Config for plucker
type Config struct {
	StatsdDigestor *digestor.Config `toml:"statsd-digestor"`
	KinesisWorker  *kinesis.Config  `toml:"kinesis-worker"`
}

func main() {
	if os.Getenv("PLUCKER_DEBUG") != "" {
		log.SetLevel(log.DebugLevel)
	}
	var config *Config

	if len(os.Args) == 2 {
		config = defaultConfig()
	} else {
		var err error

		config, err = loadConfig(os.Args[1], os.Args[2])
		if err != nil {
			log.Fatal(err.Error())
		}
	}

	// Generate statsd endpoint to send to
	statsEndpoint, err := digestor.NewStatter(config.StatsdDigestor)
	if err != nil {
		log.Fatalf("Error building statter: %s", err.Error())
	}
	defer closeHelper(statsEndpoint)

	statsdDigestor, err := digestor.NewStatsdDigestor(config.StatsdDigestor)
	if err != nil {
		log.Fatalf("Error starting plucker: %s", err.Error())
	}
	statsdDigestor = statsdDigestor.WithStatter(statsEndpoint)

	handler := statsdDigestor.Handle
	worker, err := kinesis.NewWorker(config.KinesisWorker, handler)
	if err != nil {
		log.Fatalf("Error creating Kinesis Worker: %s", err.Error())
	}

	log.Print("[Event]: Starting plucker")

	wg := sync.WaitGroup{}
	wg.Add(2)
	// Spin up workers
	go func() { worker.Fetch(); wg.Done() }()
	go func() { worker.Process(); wg.Done() }()

	// Pass a signal to the pipeline when a stop is issued
	sigc := make(chan os.Signal, 1)
	signal.Notify(sigc, syscall.SIGINT, syscall.SIGTERM)

	// Wait for a stop
	<-sigc
	log.Info("received signal, shutting down")
	worker.Stop()

	wg.Wait()

	log.Info("plucker has been stopped; see you later alligator.")
}

// LoadConfig loads a toml config from file overriding any defaults
func loadConfig(configPath, statsdEventsDir string) (*Config, error) {
	config := defaultConfig()

	parsedConfig := &Config{}
	_, err := toml.DecodeFile(configPath, parsedConfig)
	if err != nil {
		return nil, fmt.Errorf("Error parsing config: %s", err.Error())
	}

	err = mergo.MergeWithOverwrite(config, parsedConfig)
	if err != nil {
		return nil, fmt.Errorf("Error merging configs: %s", err.Error())
	}

	statsdEventRules, err := loadStatsdEventConfigs(statsdEventsDir)
	if err != nil {
		return nil, fmt.Errorf("Failed to retrieve statsd event rules: %s", err.Error())
	}

	config.StatsdDigestor.EventRules = statsdEventRules
	return config, nil
}

func loadStatsdEventConfigs(configDir string) (digestor.EventRules, error) {
	var buffer bytes.Buffer
	err := filepath.Walk(configDir, func(path string, info os.FileInfo, err error) error {
		if !info.Mode().IsRegular() {
			return nil
		}

		b, err := ioutil.ReadFile(path)
		if err != nil {
			return fmt.Errorf("Error reading file %q: %s", path, err.Error())
		}

		_, err = buffer.Write(b)
		if err != nil {
			return fmt.Errorf("Error writing to buffer: %s", err.Error())
		}
		return nil
	})
	if err != nil {
		return nil, fmt.Errorf("Error reading statsd event configs: %s", err.Error())
	}

	rules := make(digestor.EventRules)
	_, err = toml.Decode(buffer.String(), &rules)
	if err != nil {
		return nil, fmt.Errorf("Error parsing config: %s", err.Error())
	}
	return rules, nil
}

// DefaultConfig to use for plucker
func defaultConfig() *Config {
	return &Config{
		KinesisWorker:  kinesis.DefaultConfig(),
		StatsdDigestor: digestor.DefaultConfig(),
	}
}

func closeHelper(closer io.Closer) {
	err := closer.Close()
	if err != nil {
		log.Printf("Error closing closer: %s\n", err.Error())
	}
}
