package main

import (
	"bytes"
	"fmt"
	"io/ioutil"
	"net/http"
	"os"
	"strconv"
	"text/template"

	"github.com/sirupsen/logrus"

	goji "goji.io"
	"goji.io/pat"
)

var staticDir string = "dist"

// The config template (basically some simple JS)
var configJSTemplate = `
window.env = {
  ENVIRONMENT: '{{.Environment}}',
  API_BASE: '{{.APIBase}}',
  HMS_BASE: '{{.HMSBase}}',
  MESSAGING_BASE: '{{.MessagingBase}}'
};
`

type Config struct {
	Environment   string
	APIBase       string
	HMSBase       string
	MessagingBase string
}

func main() {
	logrus.Info("Initializing router")
	mux := goji.NewMux()

	if redirectURL, found := os.LookupEnv("REDIRECT_URL"); found {
		mountRedirect(mux, redirectURL)
	} else {
		mountTinkerRoutes(mux)
	}

	logrus.Info("Ready to roll")
	http.ListenAndServe("0.0.0.0:8000", mux)
}

func mountRedirect(mux *goji.Mux, redirectURL string) {
	logrus.Info("Mounting redirect to ", redirectURL)
	statusCode := http.StatusFound

	if permanentRedirect, found := os.LookupEnv("PERMANENT_REDIRECT"); found {
		if s, err := strconv.ParseBool(permanentRedirect); err == nil && s {
			logrus.Info("Setting redirect to permanent")
			statusCode = http.StatusMovedPermanently
		}
	}

	mux.HandleFunc(pat.New("/services/:serviceID"), func(w http.ResponseWriter, r *http.Request) {
	  serviceID := pat.Param(r, "serviceID")

	  serviceRedirect := fmt.Sprintf("%s/services/%s/details", redirectURL, serviceID)

	  http.Redirect(w, r, serviceRedirect, statusCode)
  })

	mux.Handle(pat.New("/*"), http.RedirectHandler(redirectURL, statusCode))
}

func mountTinkerRoutes(mux *goji.Mux) {
	var uiBase = ""
	logrus.Info("Mounting standard tinker routes")
	logrus.Info("Generating config from environment variables")
	c := generateConfig()
	configJS := createConfig(c)

	logrus.Info("Loading index page into memory")
	indexData, err := ioutil.ReadFile(fmt.Sprintf("%s%s/index.html", staticDir, uiBase))
	if err != nil {
		panic(err)
	}
	indexContent := string(indexData)

	// If the client asks for the generated config JS, serve that
	logrus.Info("Loading routes")
	mux.HandleFunc(pat.Get(uiBase+"/config.js"),
		func(w http.ResponseWriter, r *http.Request) {
			fmt.Fprintf(w, configJS)
		},
	)
	// If the client wants a static asset (JS/CSS/HTML), serve it
	mux.Handle(pat.Get(uiBase+"/:file.:ext"), http.FileServer(http.Dir(staticDir)))
	mux.Handle(pat.Get(uiBase+"/styles/:file.:ext"), http.FileServer(http.Dir(staticDir)))
	mux.Handle(pat.Get(uiBase+"/fonts/:file.:ext"), http.FileServer(http.Dir(staticDir)))
	// Otherwise always serve the index page and let React handle routes/URIs
	mux.HandleFunc(pat.Get("/*"),
		func(w http.ResponseWriter, r *http.Request) {
			fmt.Fprintf(w, indexContent)
		},
	)
}

// Reads env vars and creates a config object
func generateConfig() *Config {
	// Verify that the proper environment variables are set
	var environment string
	var apiBase string
	var hmsBase string
	var messagingBase string
	var found bool
	if environment, found = os.LookupEnv("ENVIRONMENT"); !found {
		panic("ENVIRONMENT not set in environment")
	}
	if apiBase, found = os.LookupEnv("API_BASE"); !found {
		panic("API_BASE not set in environment")
	}
	if hmsBase, found = os.LookupEnv("HMS_BASE"); !found {
		panic("HMS_BASE not set in environment")
	}
	if messagingBase, found = os.LookupEnv("MESSAGING_BASE"); !found {
		panic("MESSAGING_BASE not set in environment")
	}
	// Create a config
	config := &Config{
		Environment:   environment,
		APIBase:       apiBase,
		HMSBase:       hmsBase,
		MessagingBase: messagingBase,
	}
	return config
}

// Takes a config object and templates the JS to hand to clients
func createConfig(config *Config) string {
	// create config
	logrus.Info("Creating config")
	t := template.New("config")
	t, err := t.Parse(configJSTemplate)
	if err != nil {
		panic(err)
	}
	var c bytes.Buffer
	t.Execute(&c, config)
	configJS := c.String()
	return configJS
}
