// @format
import React from "react";
import {
  Glyphicon,
  Modal,
  Button,
  Label,
  Row,
  FormGroup,
  ControlLabel,
  FormControl,
  Col,
  Alert,
  Panel,
  Table,
} from "react-bootstrap";
import { utcFormat } from "d3";
import moment from "moment";
import Select from "react-select";
import WrapperTooltip from "components/WrapperTooltip";
import LineGraph from "common/LineGraph/components/LineGraph";

const queryTypes = [
  { value: "error_rate", label: "Error Rate" },
  { value: "error_count", label: "Error Count" },
  { value: "request_count", label: "Request Count" },
  { value: "threshold_query", label: "Threshold Query" },
  { value: "percentile_50", label: "50th Percentile" },
  { value: "percentile_90", label: "90th Percentile" },
  { value: "percentile_99", label: "99th Percentile" },
];
const previewURLBase =
  "https://graphite-web.internal.justin.tv/render?width=800&height=450&target=";
const aggrigationMethods = [
  { value: "sum", label: "Sum" },
  { value: "average", label: "Average" },
  { value: "max", label: "Max" },
];
function formatTestQueryData(data) {
  // Transform dates into a useful format
  return data.map(function(d) {
    return {
      date: moment.utc(d[1] * 1000),
      value: d[0],
    };
  });
}
function formatTestQueryToolTipText(d) {
  var formatDate = utcFormat("%Y-%m-%dT%H:%M");
  return `${formatDate(d.date)}<br/>Value:${d.value}`;
}
var formatTestQueryDataFuncArgs = {};
function mergeQuries(queries, query) {
  if (query.id) {
    for (var i = 0; i < queries.length; i++) {
      if (queries[i].id === query.id) {
        queries[i] = query;
      }
    }
  } else {
    queries.push(query);
  }
  return [...queries];
}
function deleteQuery(queries, query) {
  var new_queries = [];
  if (query.id) {
    for (var i = 0; i < queries.length; i++) {
      if (queries[i].id != query.id) {
        new_queries.push(queries[i]);
      }
    }
  } else {
    for (var i = 0; i < queries.length; i++) {
      if (queries[i].id || queries[i] !== query) {
        new_queries.push(queries[i]);
      }
    }
  }
  return new_queries;
}
function valueToLabel(items, value, defaultValue) {
  var label = items.find(item => {
    return item.value === value;
  });
  if (label) {
    return label.label;
  }
  return defaultValue;
}
class FormQuerySelect extends React.Component {
  componentWillMount() {
    this.props.toggleModal(false);
  }
  render() {
    var props = this.props;
    var {
      toggleModal,
      input,
      isEditing,
      modalEnabled,
      modalQuery,
      setModalQuery,
      testQuery,
      queryTested,
      queryTestSuccess,
      queryTestError,
      queryTestData,
      setQueryTested,
    } = props;
    var value = input.value || [];
    if (isEditing) {
      var title = <ControlLabel>Queries</ControlLabel>;
      return (
        <div>
          <Modal
            bsSize="lg"
            show={modalEnabled}
            onHide={() => props.toggleModal(false)}
          >
            <Modal.Header closeButton>
              <Modal.Title>Edit Query</Modal.Title>
            </Modal.Header>
            <Modal.Body>
              <FormGroup>
                <ControlLabel>
                  <WrapperTooltip title="Query" />
                </ControlLabel>
                <FormControl
                  type="text"
                  value={modalQuery.query}
                  onChange={event => {
                    modalQuery.query = event.target.value;
                    setModalQuery(modalQuery);
                    setQueryTested(false);
                  }}
                />
              </FormGroup>
              <ControlLabel>
                <WrapperTooltip title="Query Type" />
              </ControlLabel>
              <Select
                value={modalQuery.query_type}
                options={queryTypes}
                clearable={false}
                onChange={value => {
                  modalQuery.query_type = value.value;
                  setModalQuery(modalQuery);
                }}
              />
              <ControlLabel>
                <WrapperTooltip title="Aggregation Method" />
              </ControlLabel>
              <Select
                value={modalQuery.aggregate_type}
                options={aggrigationMethods}
                clearable={false}
                onChange={value => {
                  modalQuery.aggregate_type = value.value;
                  setModalQuery(modalQuery);
                }}
              />
              <br />
              {!queryTested && (
                <Alert bsStyle="warning">
                  You must test this query before saving
                </Alert>
              )}
              {queryTested &&
                !queryTestSuccess && (
                  <Alert bsStyle="danger">Error: {queryTestError}</Alert>
                )}
              {queryTested &&
                queryTestSuccess && (
                  <div>
                    <LineGraph
                      data={queryTestData}
                      dataFunc={formatTestQueryData}
                      dataFuncArgs={formatTestQueryDataFuncArgs}
                      toolTipTextFunc={formatTestQueryToolTipText}
                      brush={false}
                      height={200}
                      width={800}
                      clipping={false}
                    />
                    <Alert bsStyle="success">Query tested successfully</Alert>
                  </div>
                )}
            </Modal.Body>
            <Modal.Footer>
              <Button onClick={() => testQuery(modalQuery.query)}>Test</Button>
              <Button
                onClick={() => {
                  input.onChange(mergeQuries(value, modalQuery));
                  toggleModal(false);
                }}
                disabled={!(queryTested && queryTestSuccess)}
                bsStyle="primary"
              >
                Save
              </Button>
              <Button
                onClick={() => {
                  toggleModal(false);
                }}
                bsStyle="danger"
              >
                Cancel
              </Button>
            </Modal.Footer>
          </Modal>
          <FormGroup
            controlId="formControlsSelect"
            validationState={this.props.meta.error && "error"}
          >
            <Table responsive hover bordered>
              <thead>
                <tr>
                  <th>Delete</th>
                  <th>
                    <WrapperTooltip title="Edit" tooltipID="queryEditColumn" />
                  </th>
                  <th>
                    <WrapperTooltip title="Query" />
                  </th>
                  <th>
                    <WrapperTooltip title="Query Type" />
                  </th>
                  <th>
                    <WrapperTooltip title="Aggregation Method" />
                  </th>
                </tr>
              </thead>
              <tbody>
                {value.map(function(query) {
                  return (
                    <tr>
                      <td>
                        <Button
                          bsSize="small"
                          onClick={() => {
                            input.onChange(deleteQuery(value, query));
                          }}
                          bsStyle="danger"
                        >
                          <Glyphicon glyph="remove" />
                        </Button>
                      </td>
                      <td>
                        <Button
                          bsSize="small"
                          onClick={() => {
                            setModalQuery(query);
                            toggleModal(true);
                          }}
                          bsStyle="primary"
                          disabled={!query.id}
                        >
                          <Glyphicon glyph="edit" />
                        </Button>
                      </td>
                      <td>{query.query}</td>
                      <td>{valueToLabel(queryTypes, query.query_type, "")}</td>
                      <td>
                        {valueToLabel(
                          aggrigationMethods,
                          query.aggregate_type,
                          "Sum"
                        )}
                      </td>
                    </tr>
                  );
                })}
              </tbody>
            </Table>
            {this.props.meta.error && (
              <Alert bsStyle="warning">{this.props.meta.error}</Alert>
            )}
            <Row>
              <Col md={4}>
                <Button
                  onClick={() => {
                    setModalQuery({ aggregate_type: "sum" });
                    toggleModal(true);
                  }}
                  bsStyle="primary"
                >
                  Add Query
                </Button>
              </Col>
            </Row>
          </FormGroup>
        </div>
      );
    } else {
      var title = <h1>Queries</h1>;
      return (
        <Table responsive hover bordered>
          <thead>
            <tr>
              <th>
                <WrapperTooltip title="Query" />
              </th>
              <th>
                <WrapperTooltip title="Query Type" />
              </th>
              <th>
                <WrapperTooltip title="Aggregation Method" />
              </th>
            </tr>
          </thead>
          <tbody>
            {value.map(function(query) {
              return (
                <tr>
                  <td>{query.query}</td>
                  <td>{valueToLabel(queryTypes, query.query_type, "")}</td>
                  <td>
                    {valueToLabel(
                      aggrigationMethods,
                      query.aggregate_type,
                      "Sum"
                    )}
                  </td>
                </tr>
              );
            })}
          </tbody>
        </Table>
      );
    }
  }
}

export default FormQuerySelect;
