import React from "react";
import {
  Alert,
  Modal,
  Button,
  Row,
  ControlLabel,
  Col,
  FormControl,
  FormGroup,
  Label,
} from "react-bootstrap";
import Select from "react-select";
import WrapperTooltip from "components/WrapperTooltip";
import { Field } from "redux-form";
import FormTextArea from "common/FormFields/components/FormTextArea";

// These data types take a description
// "Other UGC", "Other Commercial Data", "Other User Behavior"
const otherDataTypes = new Set(["46", "47", "48", "49"]);
// This data type takes a label and description
const customDataType = "49";

class FormUserDataSelect extends React.Component {
  constructor(props, context) {
    super(props, context);
    this.handleSelectChange = this.handleSelectChange.bind(this);
    this.handleAddUserDataTypeClick = this.handleAddUserDataTypeClick.bind(
      this
    );
  }

  componentWillMount() {
    const { fetchOptions, toggleModal } = this.props;
    fetchOptions();
    toggleModal(false);
  }

  render() {
    const {
      modalEnabled,
      toggleModal,
      dataType,
      dataTypeOptions,
      changeUserDataType,
      description,
      changeDescription,
      isEditing,
      input,
    } = this.props;

    const selectedValues = this.transformValueForSelect(input.value);
    const dataTypeOptionsFormatted = this.transformValueForSelect(
      dataTypeOptions,
      true
    );
    const availableOptions = this.removeSelectedOptions(
      dataTypeOptionsFormatted,
      selectedValues
    );

    const isCustomType = otherDataTypes.has(dataType.id);

    if (isEditing) {
      return (
        <React.Fragment>
          <Modal
            bsSize="lg"
            show={modalEnabled}
            onHide={() => toggleModal(false)}
          >
            <Modal.Header closeButton>
              <Modal.Title>Add a User Data Type</Modal.Title>
            </Modal.Header>
            <Modal.Body>
              <FormGroup>
                <ControlLabel>
                  <WrapperTooltip title="User Data Type" />
                </ControlLabel>
                <Select
                  name={"dataType"}
                  options={availableOptions}
                  value={dataType}
                  onChange={changeUserDataType}
                />
              </FormGroup>
              {isCustomType && (
                <Field
                  name="custom_tag_description"
                  component={FormTextArea}
                  type="text"
                  title="Please Provide a Description Below (Required)"
                  onChange={changeDescription}
                  isEditing
                  maxLength={100}
                />
              )}
            </Modal.Body>
            <Modal.Footer>
              <Button
                bsStyle="primary"
                onClick={this.handleAddUserDataTypeClick}
                disabled={isCustomType ? !description : !dataType}
              >
                Add User Data Type
              </Button>
            </Modal.Footer>
          </Modal>
          <FormGroup
            controlId="formControlsSelect"
            validationState={this.props.meta.error && "error"}
          >
            <ControlLabel>
              <WrapperTooltip title="What Type of User Personal Data Does Your Service Store?" />
            </ControlLabel>
            <Select
              name="DataTypes"
              options={availableOptions}
              value={selectedValues}
              multi
              onChange={this.handleSelectChange}
              valueRenderer={value => (
                <Button bsStyle="link" bsSize="small">
                  {value.label}
                </Button>
              )}
            />
            <FormControl.Feedback />
            {this.props.meta.error && (
              <Alert bsStyle="warning">{this.props.meta.error}</Alert>
            )}
          </FormGroup>
          <Row>
            <Col md={4}>
              <Button
                onClick={() => {
                  toggleModal(true);
                }}
                bsStyle="primary"
              >
                Add User Data Type
              </Button>
            </Col>
          </Row>
        </React.Fragment>
      );
    } else {
      return (
        <React.Fragment>
          <p>
            <b>
              <WrapperTooltip title="What Type(s) of User Personal Data Does Your Service Store? (List all)" />
            </b>
          </p>
          <div className="service-detail-tags">
            {input.value &&
              input.value.map(tag => {
                return (
                  <Label
                    key={tag.label}
                    bsStyle="primary"
                    className="service-detail-tag"
                  >
                    {this.formatTag(tag)}
                  </Label>
                );
              })}
          </div>
        </React.Fragment>
      );
    }
  }

  // handles dropdown select and the delete buttons
  handleSelectChange(values) {
    const { input, toggleModal } = this.props;

    const selectedValues = this.transformValueForSelect(input.value);
    const mappedValues = values.map(val => val.value);
    // adding data type
    if (mappedValues.length > selectedValues.length) {
      const addedDataType = mappedValues.filter(
        val => !selectedValues.some(data => data.value.id === val.id)
      )[0];

      if (otherDataTypes.has(addedDataType.id)) {
        toggleModal(true, addedDataType);
      } else {
        input.onChange({ attr: mappedValues, action: "update" });
      }
    } else {
      input.onChange({ attr: mappedValues, action: "update" });
    }
  }

  // From the modal
  handleAddUserDataTypeClick() {
    const { dataType, toggleModal, description } = this.props;
    let newDataType = {
      ...dataType,
    };

    if (otherDataTypes.has(dataType.id)) {
      newDataType = {
        label: dataType.label,
        description,
        parent_type: {
          id: dataType.id,
          label: dataType.label,
        },
      };
    }

    toggleModal(false);
    this.addDataType(newDataType);
  }

  addDataType(tag) {
    const { input } = this.props;
    input.onChange({ attr: tag, action: "add" });
  }

  removeSelectedOptions(dataTypeOptionsFormatted, selectedValues) {
    return dataTypeOptionsFormatted.filter(
      option => !selectedValues.some(type => type.value.id === option.value.id)
    );
  }

  transformValueForSelect(tags, addInfo) {
    return (
      (tags &&
        tags.map(tag => {
          return {
            // Displayed for each selected pill
            label: `${this.formatTag(tag)}${
              addInfo && otherDataTypes.has(tag.id) ? " (Please Specify)" : ""
            }`,
            value: {
              // Displayed for the selected value in the dropdown only.
              label: tag.label,
              id: tag.id,
              description: tag.description,
              parent_type: tag.parent_type,
            },
          };
        })) ||
      []
    );
  }

  formatTag(tag) {
    if (tag.parent_type && tag.parent_type.id === customDataType) {
      return tag.description;
    }

    if (tag.parent_type && otherDataTypes.has(tag.parent_type.id)) {
      return `${tag.parent_type.label}: ${tag.description}`;
    }

    return tag.label;
  }
}

export default FormUserDataSelect;
