// @format
import { GraphqlQueryRaw } from "lib/graphql";

// ------------------------------------
// Constants
// ------------------------------------
export const FORM_QUERY_SELECT_OPTION_SELECTED =
  "FORM_QUERY_SELECT_OPTION_SELECTED";
export const FORM_QUERY_SELECT_MODAL_TOGGLED =
  "FORM_QUERY_SELECT_MODAL_TOGGLED";
export const FORM_QUERY_SELECT_MODAL_QUERY_SET =
  "FORM_QUERY_SELECT_MODAL_QUERY_SET";
export const FORM_QUERY_SELECT_MODAL_TEST_QUERY =
  "FORM_QUERY_SELECT_MODAL_TEST_QUERY";
export const FORM_QUERY_SELECT_MODAL_TEST_QUERY_SUCCESS =
  "FORM_QUERY_SELECT_MODAL_TEST_QUERY_SUCCESS";
export const FORM_QUERY_SELECT_MODAL_SET_QUERY_TESTED =
  "FORM_QUERY_SELECT_MODAL_SET_QUERY_TESTED";

// ------------------------------------
// Actions
// ------------------------------------
export function formQuerySelectQueryDeleted(id, option) {
  return {
    type: FORM_QUERY_SELECT_QUERY_DELETED,
    id: id,
    option: option,
  };
}

export function formQuerySelectModalToggled(id, modalEnabled) {
  return {
    type: FORM_QUERY_SELECT_MODAL_TOGGLED,
    modalEnabled: modalEnabled,
    id: id,
  };
}

export function formQuerySelectModalQuerySet(id, query) {
  return {
    type: FORM_QUERY_SELECT_MODAL_QUERY_SET,
    id: id,
    query: query,
  };
}
export function formQuerySelectModalSetQueryTested(id, bool) {
  return {
    type: FORM_QUERY_SELECT_MODAL_SET_QUERY_TESTED,
    id: id,
    queryTested: bool,
  };
}

export function formQuerySelectModalTestQuery(id, query) {
  // Call the API to test the query
  // Check the return, set queryTestSuccess flag
  // and queryTested flags
  return dispatch => {
    const previewURLBase =
      "https://graphite-web.internal.justin.tv/render?format=json&target=";
    // Try fetching the data, make sure it is sane
    fetch(previewURLBase + query)
      .then(function(resp) {
        if (resp.ok) {
          return resp.json();
        } else {
          throw new Error("query syntax error");
        }
      })
      .then(function(data) {
        if (data.length != 1) {
          throw new Error(
            "query result must contain exactly 1 timeseries, found " +
              data.length
          );
        } else {
          dispatch(
            formQuerySelectModalTestQuerySuccess(
              id,
              true,
              data[0].datapoints,
              null
            )
          );
        }
      })
      .catch(function(err) {
        // Hack because graphite is dumb
        // Graphite, when a query is syntactically bad,
        // doesn't hand back useful JSON formatted error info,
        // but just gives a stack trace and 500. In addition,
        // it doesn't properly set CORS headers, so the client-side
        // just sees an error message about CORS, instead of a custom
        // error message. Here we check for the CORS error and gobble it
        // and spit out 'query syntax error' to be a _bit_ more informative
        if (err.message == "Failed to fetch") {
          var msg = "query syntax error";
        } else {
          var msg = err.message;
        }
        dispatch(formQuerySelectModalTestQuerySuccess(id, false, null, msg));
      });
  };
}

function formQuerySelectModalTestQuerySuccess(id, success, data, errMsg) {
  return {
    id: id,
    type: FORM_QUERY_SELECT_MODAL_TEST_QUERY_SUCCESS,
    success: success,
    data: data,
    err: errMsg,
  };
}
// ------------------------------------
// Reducers
// ------------------------------------
export function formQuerySelect(state = {}, action) {
  switch (action.type) {
    case FORM_QUERY_SELECT_MODAL_TOGGLED:
      return {
        ...state,
        [action.id]: {
          ...state[action.id],
          modalEnabled: action.modalEnabled,
        },
      };
    case FORM_QUERY_SELECT_MODAL_QUERY_SET:
      return {
        ...state,
        [action.id]: {
          ...state[action.id],
          modalQuery: { ...action.query },
        },
      };
    case FORM_QUERY_SELECT_MODAL_TEST_QUERY_SUCCESS:
      return {
        ...state,
        [action.id]: {
          ...state[action.id],
          queryTested: true,
          queryTestSuccess: action.success,
          queryTestData: action.data,
          queryTestError: action.err,
        },
      };
    case FORM_QUERY_SELECT_MODAL_SET_QUERY_TESTED:
      return {
        ...state,
        [action.id]: {
          ...state[action.id],
          queryTested: action.queryTested,
        },
      };
    default:
      return state;
  }
}

export default formQuerySelect;
