// @format
import React from "react";
import PropTypes from "prop-types";
import Icon from "react-icons-kit";
import { caretDown } from "react-icons-kit/fa";
import { caretRight } from "react-icons-kit/fa";
import { Query } from "react-apollo";
import WrapperTooltip from "components/WrapperTooltip";
import {
  Glyphicon,
  DropdownButton,
  FormGroup,
  Form,
  FormControl,
  ControlLabel,
  Panel,
  Pagination,
  Row,
  Col,
  Table,
} from "react-bootstrap";
import {CSVLink} from "react-csv";
import Select from "react-select";
import { LoadingSpinner } from "../../../components/LoadingSpinner";
import _ from "lodash";
import gql from "graphql-tag";

const itemsPerPageOptions = [
  { value: 10, label: "10" },
  { value: 25, label: "25" },
  { value: 50, label: "50" },
  { value: 100, label: "100" },
];

function getPage(items, itemFunc, page, itemsPerPage, enablePagination) {
  // If serviceDependencies are left around after dep was deleted, this protects
  items = _.filter(items, function(i) {
    return i !== undefined && i !== null && i.downstream_service !== null;
  });
  if (enablePagination === false) {
    return itemFunc ? _.map(items, itemFunc) : items;
  }
  var l = items.length;
  let pageItems = items.slice((page - 1) * itemsPerPage, page * itemsPerPage);
  return itemFunc ? _.map(pageItems, itemFunc) : pageItems;
}

function rowData(
  index,
  item,
  columns,
) {
  return (
    <tr key={index}>
      {columns.map(function(column) {
        return resolveCell(item, column);
      })}
    </tr>
  );
}

function resolveFilters(columns, filters, data, onChange, basicFilterHeader, advancedFilterHeader, showBasicFilters, showAdvancedFilters) {
  let advancedFilterElements = [];
  let basicFilterElements = [];
  for (var i = 0; i < columns.length; i++) {
    var column = columns[i];
    var title = column.title;
    var key = column.key;
    if (column.filter === "select") {
      // try to find a formatter for this column
      var formatter = column.formatter;
      var options = iterateOptions(key, data, filters[key].value, formatter);
      if (column.isBasicFilter) {
        basicFilterElements.push(
          selectFilter(key, title, filters[key].value, options, onChange)
        );
      } else {
        advancedFilterElements.push(
          selectFilter(key, title, filters[key].value, options, onChange)
        );
      }
    }
    if (column.filter === "text") {
      if (column.isBasicFilter) {
        basicFilterElements.push(textFilter(key, title, filters[key].value, onChange));
      } else {
        advancedFilterElements.push(textFilter(key, title, filters[key].value, onChange));
      }
    }
    if (column.filter === "attributes") {
      if (column.isBasicFilter) {
        basicFilterElements.push(
          resolveAttributeFilter(data, key, title, filters[key].value, onChange)
        );
      } else {
        advancedFilterElements.push(
          resolveAttributeFilter(data, key, title, filters[key].value, onChange)
        );
      }
    }
  }
  let shouldDefaultExpand = true;
  if (document.location.href.indexOf("/serviceOwner/") !== -1) {
    // do not auto-expand basic filters on MyServices
    shouldDefaultExpand = false;
  }
  return (
    <div>
      {basicFilterElements.length > 0 &&
      <Panel defaultExpanded={shouldDefaultExpand} collapsible header={basicFilterHeader} expanded={showBasicFilters}>
        <Form
          horizontal
          onSubmit={e => {
            e.preventDefault();
          }}
        >
          {basicFilterElements}
        </Form>
      </Panel>
      }
      {advancedFilterElements.length > 0 &&
      <Panel collapsible header={advancedFilterHeader} expanded={showAdvancedFilters}>
        <Form
          horizontal
          onSubmit={e => {
            e.preventDefault();
          }}
        >
          {advancedFilterElements}
        </Form>
      </Panel>
      }
    </div>
  );
}

function resolveAttributeFilter(data, attributesKey, title, values, onChange) {
  var filterElements = [];
  var options = {};
  for (var i = 0; i < data.length; i++) {
    if (_.has(data[i], attributesKey)) {
      var attrs = _.get(data[i], attributesKey);
      for (var j = 0; j < attrs.length; j++) {
        options[attrs[j].name] = {
          label: attrs[j].name,
          value: { name: attrs[j].name, value: "" },
        };
        var val = attrs[j].name + ": " + attrs[j].value;
        options[val] = {
          label: val,
          value: { name: attrs[j].name, value: attrs[j].value },
        };
      }
    } else {
      options["Unknown"] = { label: "Unknown", value: "Unknown" };
    }
  }
  var selected = {};

  if (values != null) {
    values.forEach(function(value) {
      selected[value.value.name] = true;
    });
  }
  var res = [];
  var o = _.keys(options).sort();
  var k;
  for (k in o) {
    if (selected[options[o[k]].value.name] == null) {
      res.push(options[o[k]]);
    }
  }
  return (
    <FormGroup key={attributesKey} controlId={"formFilters-" + attributesKey}>
      <Col md={4} sm={4} componentClass={ControlLabel}>
        <WrapperTooltip title={title} tooltipID={title} />
      </Col>
      <Col md={8} sm={8}>
        <Select
          name={attributesKey}
          options={res}
          value={values}
          multi
          onChange={e => {
            if (e.length == 0) {
              onChange(attributesKey, null, null);
            } else {
              onChange(attributesKey, e, e);
            }
          }}
        />
      </Col>
    </FormGroup>
  );
}
// this needed to be a separate function so that the scope of
// `name` would always be correct instead of always the value
// of the last filter registered
function selectFilter(name, title, filterValue, options, onChange) {
  return (
    <FormGroup key={`select-${name}`} controlId={"formFilters-" + name}>
      <Col md={4} sm={4} componentClass={ControlLabel}>
        <WrapperTooltip title={title} tooltipID={title} />
      </Col>
      <Col md={8} sm={8}>
        <Select
          options={options}
          value={filterValue}
          onChange={e => {
            if (e === null) {
              onChange(name, null, null);
            } else {
              onChange(name, e.value, e.value);
            }
          }}
        />
      </Col>
    </FormGroup>
  );
}

function textFilter(name, title, filterValue, onChange) {
  return (
    <FormGroup key={`text-${name}`} controlId={"formFilters-" + name}>
      <Col md={4} sm={4} componentClass={ControlLabel}>
        <WrapperTooltip title={title} tooltipID={title} />
      </Col>
      <Col md={8} sm={8}>
        <FormControl
          componentClass="input"
          type="text"
          value={filterValue === null ? "" : filterValue}
          placeholder="Start typing to search..."
          onChange={e => {
            if (e.target.value === "") {
              onChange(name, null, null);
            } else {
              onChange(name, e.target.value, e.target.value);
            }
          }}
        />
      </Col>
    </FormGroup>
  );
}

// Takes all the data and generates the available
// options for a dropdown select filter
function iterateOptions(key, data, current, formatter) {
  var options = {};
  if (current !== null) {
    options[current] = { label: current, value: current };
  }
  for (var i = 0; i < data.length; i++) {
    // If this select filter uses a formatter, take the `value`
    // portion of the formatted column (as opposed to `display`)
    if (data[i] === undefined || data[i] === null) {
      continue;
    }
    if (formatter) {
      var val = formatter(data[i]).value;
      options[val] = { label: val, value: val };
    } else if (_.has(data[i], key)) {
      var val = _.get(data[i], key);
      options[val] = { label: val, value: val };
    } else {
      options["Unknown"] = { label: "Unknown", value: "Unknown" };
    }
  }
  var res = [];
  var o = _.keys(options).sort();
  var k;
  for (k in o) {
    res.push(options[o[k]]);
  }
  return res;
}

function resolveCell(item, column) {
  if (column.filter === "attributes") {
    return;
  }
  return <td key={column.key}>{getCellData(item, column)}</td>
}

function getCellData(item, column, valueOnly=false) {
  if (_.has(column, "formatter") && _.isFunction(column.formatter)) {
    if (!valueOnly) {
      return column.formatter(item).display;
    } else {
      return column.formatter(item).value;
    }
  } else if (_.has(item, column.key)) {
    return _.get(item, column.key);
  } else {
    return `Missing ${column.key}`
  }
}

function columnHeader(column, sorting, changeSorting) {
  if (column.filter === "attributes") {
    return;
  }
  if (sorting.order === "desc") {
    var glyph = <Glyphicon glyph="arrow-down" />;
  } else {
    var glyph = <Glyphicon glyph="arrow-up" />;
  }
  if (column.key === sorting.key) {
    return (
      <th
        key={column.key}
        onClick={e => changeSorting(column.key)}
        style={{ cursor: "pointer" }}
      >
        <WrapperTooltip title={column.title} /> {glyph}
      </th>
    );
  } else {
    return (
      <th
        key={column.key}
        onClick={e => changeSorting(column.key)}
        style={{ cursor: "pointer" }}
      >
        <WrapperTooltip title={column.title} />
      </th>
    );
  }
}

function getTableData(
  items,
  columns,
  asObject,
) {
  let tableData = []
  for (let item of items) {
    let rowData = [];
    if (asObject) {
      rowData = columns.reduce(function(map, column) {
        map[column.key] = getCellData(item, column, true)
        return map;
      }, {});
    } else {
      rowData = columns.map(function(column) {
        return getCellData(item, column, true);
      });
    }
    tableData = tableData.concat(rowData);
  }
  return tableData
}

function downloadAsCSVData(
  items,
  columns
) {
  columns = columns.filter((c) => c.key !== "attributes");
  let column_keys = columns.map(function(column) {
    return column.key
  })
  let data = [column_keys];
  for (let item of getTableData(items, columns, true)) {
    let row_data = columns.map(function(column) {
      return item[column.key]
    });
    data = data.concat([row_data])
  }
  return data;
}

function downloadAsJSONURL(
  items,
  columns
) {
  columns = columns.filter((c) => c.key !== "attributes");
  let data = getTableData(items, columns, true);
  let json = JSON.stringify(data);
  let blob = new Blob([json], {type: "application/json"});
  let myURL = window.URL || window.webkitURL;
  return myURL.createObjectURL(blob);
}

// These allow the dropdown to close on click
const JSONDropdown = ({href, onSelect}) => {
  return <li onClick={() => onSelect()}>
    <a download={"catalogDownload.json"} href={href}>
      JSON
    </a>
  </li>
}

const CSVDropdown = ({csvData, onSelect}) => {
  return <li onClick={() => onSelect()}>
    <CSVLink data={csvData} filename={"catalogDownload.csv"}
               target={"_blank"}>CSV</CSVLink></li>
}

const fetchPrimaryOwnerDataQuery = gql`
  query fetchPrimaryOwnerData($user_ids: [ID]!) {
    contacts(user_ids: $user_ids) {
      user_id
      email
      manager_user_id
      manager_preferred_name
      team_name
      bu_name
      org_name
    }
  }
`;

var primaryOwnerData = [];
var primaryOwnerDataReady = false;

export function isPrimaryOwnerDataReady() {
  return primaryOwnerDataReady
}

export function resetPrimaryOwnerDataReady() {
  primaryOwnerDataReady = false;
}

export function getPrimaryOwnerData(user_id) {
  if (primaryOwnerData.length > 0) {
    return primaryOwnerData.find(function(contact) {
      return contact.user_id === user_id
    });
  }
}

export const PrimaryOwnerDataView = ({ action, user_ids }) => (
  <Query query={fetchPrimaryOwnerDataQuery} variables={{ user_ids }}>
    {({ loading, error, data}) => {
      if (loading) {
        return <LoadingSpinner />
      }
      if (error) {
        console.log(error);
        console.trace();
        return null
      }
      if (!data || data === undefined || data.length === 0){
        return null;
      }

      primaryOwnerData = data.contacts;

      if (!isPrimaryOwnerDataReady()) {
        primaryOwnerDataReady = true;
        action();
      }

      return <div></div>
    }}
  </Query>
)

PrimaryOwnerDataView.propTypes = {
  user_ids: PropTypes.array,
  loading: PropTypes.bool,
  error: PropTypes.object,
  action: PropTypes.func
}

export const ServiceTable = ({
  data = [],
  dataFunc,
  columns,
  isLoading,
  hasErrored,
  changePage,
  changeItemsPerPage,
  currentPage,
  itemsPerPage,
  changeFilter,
  changeSorting,
  filters,
  sorting,
  enablePagination,
  enableFilters,
  toggleFilterDropdown,
  showBasicFilters: showBasicFilters,
  showAdvancedFilters: showAdvancedFilters,
  showExports=false,
}) => {
  // Some small UI things to indicate that
  // the Filters dropdown is clickable
  var rightIcon = <Icon icon={caretRight} />;
  var downIcon = <Icon icon={caretDown} />;
  let basicFilterHeader = (
    <div
      style={{ cursor: "pointer" }}
      onClick={() => toggleFilterDropdown(!showBasicFilters)}
    >
      {showBasicFilters ? downIcon : rightIcon}
      Basic Filters
    </div>
  );
  let advancedFilterHeader = (
    <div
      style={{ cursor: "pointer" }}
      onClick={() => toggleFilterDropdown(!showAdvancedFilters)}
    >
      {showAdvancedFilters ? downIcon : rightIcon}
      Hierarchical Filters (based on Primary Owner)
    </div>
  );
  let tableHeader = null;
  if (enableFilters || showExports) {
    tableHeader = (<Row>
      <Col md={6}>
            {enableFilters &&
            (_.keys(filters).length > 0 &&
                <div>
                  {resolveFilters(columns, filters, data, changeFilter, basicFilterHeader, advancedFilterHeader, showBasicFilters, showAdvancedFilters)}
                </div>
              )
            }
      </Col>
      <Col md={2} mdOffset={4}>
            {showExports &&
            <div className={'pull-right'}>
              <DropdownButton
                bsStyle={'primary'}
                title={'Export As'}
                id={'export-button'}
              >
                <JSONDropdown href={downloadAsJSONURL(data, columns)} />
                <CSVDropdown csvData={downloadAsCSVData(data, columns)} />
              </DropdownButton>
            </div>
        }
      </Col>
          </Row>)

  }
  return (
    <Row>
      <Col md={12}>
        {tableHeader}
        <Row>
          <Col md={12}>
            <Table hover bordered striped>
              <thead>
                <tr>
                  {columns.map(function(column) {
                    return columnHeader(column, sorting, changeSorting);
                  })}
                </tr>
              </thead>
              <tbody>
                {getPage(
                  data,
                  dataFunc,
                  currentPage,
                  itemsPerPage,
                  enablePagination
                ).map(function(item, index) {
                  return rowData(
                    index,
                    item,
                    columns,
                  );
                })}
              </tbody>
            </Table>
          </Col>
        </Row>
        {isLoading && (
          <Row className="show-grid">
            <Col xs={1} md={3} />
            <Col xs={4} md={6}>
              <LoadingSpinner />
            </Col>
            <Col xs={1} md={3} />
          </Row>
        )}
        {enablePagination && (
          <Row className="show-grid">
            <Col xs={10} md={10}>
              <Pagination
                prev
                next
                first
                last
                ellipsis
                boundaryLinks
                items={Math.ceil(data.length / itemsPerPage)}
                maxButtons={5}
                activePage={currentPage}
                onSelect={changePage}
              />
            </Col>
            <Col xs={2} md={2}>
              <div>Per Page</div>
              <div>
                <Select
                  name={"itemsPerPage"}
                  options={itemsPerPageOptions}
                  value={itemsPerPage}
                  clearable={false}
                  onChange={e => {
                    changeItemsPerPage(e.value);
                  }}
                />
              </div>
              <div>({data.length} total)</div>
            </Col>
          </Row>
        )}
      </Col>
    </Row>
  );
};

ServiceTable.propTypes = {
  data: PropTypes.array.isRequired,
  dataType: PropTypes.string.isRequired,
  dataFunc: PropTypes.func,
  columns: PropTypes.array.isRequired,
  isLoading: PropTypes.bool,
  hasErrored: PropTypes.bool,
};
