import React from "react";
import PropTypes from "prop-types";
import { connect } from "react-redux";
import { createSelector } from "reselect";
import _ from "lodash";
import { ServiceTable } from "../components/ServiceTable";
import { PrimaryOwnerDataView, resetPrimaryOwnerDataReady } from "../components/ServiceTable";
import moment from "moment";
import {
  tableInitAction,
  tableCurrentPageAction,
  tableItemsPerPageAction,
  tableSortingAction,
  tableFiltersAction,
  tableShowFiltersAction,
} from "../modules/service_table";

class ServiceContainer extends React.Component {
  constructor(props) {
    super(props)
    // Bind the this context to the state-handler function
    this.setPrimaryOwnerDataAvailable = this.setPrimaryOwnerDataAvailable.bind(this);
    // Set state
    this.state = {
      primaryOwnerDataAvailable: false
    };
  }
  // This method will be sent to the child component
  setPrimaryOwnerDataAvailable() {
    this.setState({
      primaryOwnerDataAvailable: true
    });
  }
  resetPrimaryOwnerDataAvailable() {
    this.setState({
      primaryOwnerDataAvailable: false
    });
  }
  componentDidMount() {
    var filterData = {};
    // iterate over all specified columns and format
    // initial states of filters
    for (var i = 0; i < this.props.columns.length; i++) {
      var column = this.props.columns[i];
      if (column.filter !== null) {
        filterData[column.key] = { value: null, type: column.filter };
      }
    }
    var sortData = {
      key: this.props.columns[0].key,
      order: "asc",
    };
    // set initial filters, basically just all empty
    this.props.tableInit(filterData, sortData, this.props.defaultItemsPerPage || 10);
    resetPrimaryOwnerDataReady();
  }
  componentDidUpdate(prevProps, prevState, prevContext) {
    if (prevState.primaryOwnerDataAvailable === true) {
      resetPrimaryOwnerDataReady();
      this.resetPrimaryOwnerDataAvailable()
    }
  }
  // Get the data that will be rendered on the current showing page
  getDataOnPage() {
    const {
      enablePagination,
      data,
      dataFunc,
      currentPage,
      itemsPerPage,
    } = this.props;

    // If no pagination, just return everything
    if (enablePagination === false) {
      return dataFunc ? _.map(data, dataFunc) : data;
    }

    let pageData = data.slice(
      (currentPage - 1) * itemsPerPage,
      currentPage * itemsPerPage
    );

    return dataFunc ? _.map(pageData, dataFunc) : pageData;
  }
  render() {
    let primaryOwnerIDs = [];
    for (let service of this.props.data) {
      if (service.primary_owner != null && !primaryOwnerIDs.includes(service.primary_owner.uid)) {
        primaryOwnerIDs.push(service.primary_owner.uid);
      }
    }
    return (
      <div>
        <PrimaryOwnerDataView action={this.setPrimaryOwnerDataAvailable} user_ids={primaryOwnerIDs}/>
        <ServiceTable
          tableId={this.props.tableId}
          data={this.props.data}
          dataType={this.props.dataType}
          dataFunc={this.props.dataFunc}
          columns={this.props.columns}
          changePage={this.props.changePage}
          changeItemsPerPage={this.props.changeItemsPerPage}
          filters={this.props.filters}
          sorting={this.props.sorting}
          modal={this.props.modal}
          modalData={this.props.modalData}
          modalItem={this.props.modalItem}
          changeFilter={this.props.changeFilter}
          changeSorting={this.props.changeSorting}
          currentPage={this.props.currentPage}
          itemsPerPage={this.props.itemsPerPage}
          enableFilters={this.props.enableFilters}
          enablePagination={this.props.enablePagination}
          toggleFilterDropdown={this.props.toggleFilterDropdown}
          showFilters={this.props.showFilters}
          showExports={this.props.showExports}
        />
      </div>
    );
  }
}

var mapDispatchToProps = function(dispatch, ownProps) {
  return {
    tableInit: (filters, sorting, defaultItemsPerPage) =>
      dispatch(tableInitAction(ownProps.tableId, filters, sorting, defaultItemsPerPage)),
    changePage: i => dispatch(tableCurrentPageAction(ownProps.tableId, i)),
    changeItemsPerPage: i =>
      dispatch(tableItemsPerPageAction(ownProps.tableId, i)),
    changeFilter: (n, v) =>
      dispatch(tableFiltersAction(ownProps.tableId, n, v)),
    changeSorting: key => dispatch(tableSortingAction(ownProps.tableId, key)),
    toggleFilterDropdown: bool =>
      dispatch(tableShowFiltersAction(ownProps.tableId, bool)),
  };
};

function hasAttribute(name, value, attrs) {
  for (var j = 0; j < attrs.length; j++) {
    var val = attrs[j].name + ": " + attrs[j].value;
    if (attrs[j].name === name)
      if (value === "") {
        return true;
      }
    if (attrs[j].value === value) {
      return true;
    }
  }
  return false;
}

// Filter data using the filter state and the 'reselect'
// library
export const getFilters = (state, props) => state.filters;
export const getData = (state, props) => props.data;
export const getDataFunc = (state, props) => props.dataFunc;
export const getSorting = (state, props) => state.sorting;
export const getFormatters = (state, props) => {
  var formatterMap = {};
  for (var i = 0; i < props.columns.length; i++) {
    var col = props.columns[i];
    if (_.has(col, "formatter")) {
      formatterMap[col.key] = col.formatter;
    }
  }
  return formatterMap;
};

export const makeGetFilteredData = () => {
  return createSelector(
    [getData, getFilters, getSorting, getFormatters, getDataFunc],
    (allData, filters, sorting, formatters, dataFunc) => {
      var filterKeys = _.keys(filters);
      var filteredData = allData || [];
      // first apply filters
      for (var i = 0; i < _.keys(filters).length; i++) {
        var filter = filters[filterKeys[i]];
        // only apply a filter of the filter has a selected value
        if (filter.value !== null) {
          filteredData = filteredData.filter(function(d) {
            if (filter.type === "select") {
              if (filter.value == "Unknown") {
                // special case for finding `null`s
                return !_.has(d, filterKeys[i]);
              } else if (_.has(formatters, filterKeys[i])) {
                return formatters[filterKeys[i]](d).value === filter.value;
              } else {
                return _.get(d, filterKeys[i]) === filter.value;
              }
            } else if (filter.type === "text") {
              return _.get(d, filterKeys[i])
                .toLowerCase()
                .includes(filter.value.toLowerCase());
            } else if (filter.type === "attributes") {
              var attrs = _.get(d, filterKeys[i]);
              for (var j = 0; j < filter.value.length; j++) {
                var value = filter.value[0].value;
                if (!hasAttribute(value.name, value.value, attrs)) {
                  return false;
                }
              }
              return true;
            }
          });
        }
      }
      // now sort the data based on the sorting config
      if (sorting.key === null) {
        return filteredData;
      }
      // sort according to the key, filtering
      // null values down to the bottom
      var sortedData = filteredData.slice(0).sort(function(a, b) {
        // Use a formatter value if a formatter function is specified
        if (_.has(formatters, sorting.key)) {
          var getVal = function(x) {
            return _.get(formatters, sorting.key)(x).value;
          };
        } else {
          var getVal = function(x) {
            return _.get(x, sorting.key);
          };
        }

        // Apply dataFunc if present before extracting data.
        if (dataFunc) {
          var aValue = String(getVal(dataFunc(a)));
          var bValue = String(getVal(dataFunc(b)));
        } else {
          var aValue = String(getVal(a));
          var bValue = String(getVal(b));
        }
        if (aValue === null && bValue === null) {
          return 0;
        } else if (aValue === null) {
          return -1;
        } else if (bValue === null) {
          return 1;
        } else if (aValue.toLowerCase() > bValue.toLowerCase()) {
          return 1;
        } else if (bValue.toLowerCase() > aValue.toLowerCase()) {
          return -1;
        } else {
          return 0;
        }
      });
      if (sorting.order === "asc") {
        return sortedData;
      } else {
        return sortedData.slice().reverse();
      }
    }
  );
};

export const getItemData = (state, props) => props.data;
export const getModal = state => state.modal;
// If the modal is enabled, we need to find the
// item in the dataset which the modal is enabled
// for. The modal needs to show some basic overview
// information for the item, so we use a selector
// to take the modal state and find the item (based on ID)
// and inject that item directly into the ServiceTable
// props under a special name
//
// If the modal isnt enabled we can just return null
export const makeGetModalItem = () => {
  return createSelector([getItemData, getModal], (allItems, modal) => {
    // if the modal isnt enabled, return null
    if (modal.enabled === false) {
      return null;
    }
    // derive itemData
    for (var i = 0; i < allItems.length; i++) {
      if (allItems[i].id === modal.id) {
        var item = allItems[i];
        break;
      }
    }
    return item;
  });
};

// Fancy footwork to get the Table to memoize
// filtering data for every instance of the table
// through tableId lookups.
var makeMapStateToProps = () => {
  const getFilteredData = makeGetFilteredData();
  const mapStateToProps = function(state, ownProps) {
    // If there is no table state for this table, hand over a sane default
    // until the TABLE_INIT action is invoked on component mount
    var defaultTableData = {
      filters: {},
      pagination: {
        currentPage: 1,
        itemsPerPage: 10,
      },
      sorting: {
        key: null,
        order: "asc",
      },
      modal: {
        enabled: false,
        loading: false,
        id: null,
        window: null,
        data: null,
      },
      modalItem: null,
      showFilters: false,
    };
    if (!_.has(state.availability_table, ownProps.tableId)) {
      var tableData = defaultTableData;
      var data = [];
      var modalItem = null;
      var modalData = null;
    } else {
      var tableData = state.availability_table[ownProps.tableId];
      var data = getFilteredData(
        state.availability_table[ownProps.tableId],
        ownProps
      );
    }
    if (_.has(ownProps, "enablePagination")) {
      var enablePagination = ownProps.enablePagination;
    } else {
      var enablePagination = true;
    }
    if (_.has(ownProps, "enableFilters")) {
      var enableFilters = ownProps.enableFilters;
    } else {
      var enableFilters = true;
    }
    return {
      data: data,
      filters: tableData.filters,
      sorting: tableData.sorting,
      currentPage: tableData.pagination.currentPage,
      itemsPerPage: tableData.pagination.itemsPerPage,
      modal: tableData.modal,
      modalItem: modalItem,
      modalData: modalData,
      enablePagination: enablePagination,
      enableFilters: enableFilters,
      showFilters: tableData.showFilters,
    };
  };
  return mapStateToProps;
};

ServiceContainer.propTypes = {
  data: PropTypes.array.isRequired,
  dataType: PropTypes.string.isRequired,
  dataFunc: PropTypes.func,
  defaultItemsPerPage: PropTypes.number,
};

export default connect(makeMapStateToProps, mapDispatchToProps)(
  ServiceContainer
);
