import moment from "moment";

import { graphqlMutationPromise, graphqlQueryPromise } from "lib/graphql";
import gql from "graphql-tag";

// Service types
export const REQUEST_SERVICE = "REQUEST_SERVICE";
export const REQUEST_SERVICES = "REQUEST_SERVICES";
export const RECEIVE_SERVICE = "RECEIVE_SERVICE";
export const RECEIVE_SERVICES = "RECEIVE_SERVICES";
export const UPDATE_SERVICE = "UPDATE_SERVICE";
export const DELETE_SERVICE = "DELETE_SERVICE";
export const DELETE_SERVICE_SUCCESS = "DELETE_SERVICE_SUCCESS";

export const serviceShallowDataFragment = gql`
  fragment serviceShallowData on Service {
    id
    name
    state
    primary_owner {
      uid
      slack {
        id
        name
      }
      cn
      preferred_name
      inactive
    }
    type
    endpoints {
      id
      service_id
      region
      stage
      vpc_endpoint_service_id
      hostname
      port
      ssl_enabled
    }
    latest_service_audits {
      id
      audit_type
      auditor
      action
      audit_time
      audit_value
    }
    attributes {
      id
      name
      value
    }
  }
`;

const serviceDataFragment = gql`
  fragment serviceData on Service {
    id
    name
    state
    stores_user_data
    access_requests
    access_exception
    access_exception_validated_by
    deletion_requests
    deletion_exception
    deletion_exception_validated_by
    user_data_types {
      id
      label
      description
      parent_type {
        id
        label
      }
    }
    primary_owner_id
    primary_owner {
      employee_number
      cn
      uid
      amazon_uid
      department
      manager
      building_name
      preferred_name
      inactive
      slack {
        id
        name
      }
      status
      error
    }
    attributes {
      id
      name
      value
    }
    description
    pagerduty
    slack_channel {
      id
      name
    }
    slack_channel_id
    type
    endpoints {
      id
      service_id
      region
      stage
      vpc_endpoint_service_id
      hostname
      port
      ssl_enabled
    }
    service_upstreams {
      id
      root_service {
        ...serviceShallowData
      }
      downstream_service {
        ...serviceShallowData
      }
    }
    service_downstreams {
      id
      root_service {
        ...serviceShallowData
      }
      downstream_service {
        ...serviceShallowData
      }
    }
    latest_service_audits {
      id
      audit_type
      auditor
      action
      audit_time
      audit_value
    }
    service_audits {
      id
      audit_type
      auditor
      action
      audit_time
      audit_value
    }
    logs(limit: 5) {
      id
      item_type
      item_id
      item_label
      action
      author
      created
      before
      after
    }
  }
  ${serviceShallowDataFragment}
`;

const fetchServiceQuery = gql`
  query FetchService($id: ID!) {
    service(id: $id) {
      ...serviceData
    }
  }
  ${serviceDataFragment}
`;

const fetchAllServicesQuery = gql`
  query FetchAllServices {
    services {
      ...serviceShallowData
    }
  }
  ${serviceShallowDataFragment}
`;

const createServiceQuery = gql`
  mutation CreateService($service: ServiceInput!) {
    service: createService(service: $service) {
      ...serviceData
    }
  }
  ${serviceDataFragment}
`;

// Update a service, get all the data back so that
// we can refresh the view with the updated information
export const updateServiceQuery = gql`
  mutation UpdateServiceByID($id: ID!, $service: ServiceInput!) {
    service: updateService(id: $id, service: $service) {
      ...serviceData
    }
  }
  ${serviceDataFragment}
`;

const deleteServiceQuery = gql`
  mutation DeleteServiceByID($id: ID!) {
    deleteService(id: $id) {
      id
    }
  }
`;

// Create a service audit
const createServiceAuditQuery = gql`
  mutation CreateServiceAudit($service_audit: ServiceAuditInput!) {
    service_audit: createServiceAudit(service_audit: $service_audit) {
      id
      audit_type
      auditor
      action
      audit_time
      audit_value
    }
  }
`;

//
// Actions
//

function requestService(serviceID) {
  return {
    type: REQUEST_SERVICE,
    serviceID: serviceID,
  };
}

function receiveService(serviceID, service, status) {
  return {
    type: RECEIVE_SERVICE,
    serviceID: serviceID,
    service: service,
    status: status,
    receivedAt: Date.now(),
    detailed: true,
  };
}

function requestServices() {
  return {
    type: REQUEST_SERVICES,
  };
}

// Receive many services with minimal details.
// Useful for index pages, etc
function receiveServices(services, status) {
  return {
    type: RECEIVE_SERVICES,
    services: services,
    status: status,
    receivedAt: Date.now(),
    detailed: false,
  };
}

function updateService(serviceID, service, status) {
  return {
    type: UPDATE_SERVICE,
    serviceID: serviceID,
    service: service,
    status: status,
    receivedAt: Date.now(),
  };
}

function deleteService(serviceID) {
  return {
    type: DELETE_SERVICE,
    serviceID: serviceID,
    receivedAt: Date.now(),
  };
}

export function fetchService(serviceID) {
  return dispatch => {
    // Inform that the action is starting
    dispatch(requestService(serviceID));

    let variables = {
      id: serviceID,
    };

    return graphqlQueryPromise({
      query: fetchServiceQuery,
      variables: variables,
    })
      .then(response => {
        return dispatch(receiveService(serviceID, response.data.service, {}));
      })
      .catch(error => {
        console.log(error);
        console.trace();
        dispatch(receiveService(serviceID, undefined, {error: error}));
        throw error;
      });
  };
}

export function fetchAllServices() {
  return dispatch => {
    dispatch(requestServices());
    graphqlQueryPromise({
      query: fetchAllServicesQuery,
      variables: {},
    })
      .then(response => {
        return dispatch(receiveServices(response.data.services, {}));
      })
      .catch(error => {
        return dispatch(receiveServices([], { error: error }));
      });
  };
}

function shouldFetchService(state, serviceID) {
  const service = state.servicesByID[serviceID];
  if (!service) {
    return true;
  } else if (service.isFetching) {
    return false;
  } else if (!service.detailed) {
    return true;
  }
}

export function fetchServiceIfNeeded(serviceID) {
  // FIXME: This bugs me that I am deferencing .goracle here...
  return (dispatch, getState) => {
    if (shouldFetchService(getState().goracle, serviceID)) {
      return dispatch(fetchService(serviceID));
    } else {
      return new Promise(function (resolve, reject) {
        resolve(getState().goracle.servicesByID[serviceID]);
      });
    }
  };
}

export function createService(serviceData) {
  return dispatch => {
    let variables = {
      service: serviceData,
    };

    return graphqlMutationPromise({
      mutation: createServiceQuery,
      variables: variables,
    })
      .then(function (response) {
        dispatch(
          receiveService(response.data.service.id, response.data.service, {})
        );
        return response.data.service;
      })
      .catch(function (error) {
        dispatch(receiveService("new", serviceData, {error: error}));
        throw error;
      });
  };
}

export function updateService(serviceData) {
  return dispatch => {
    let variables = {
      id: serviceData.id,
      service: serviceData,
    };

    return graphqlMutationPromise({
      mutation: updateServiceQuery,
      variables: variables,
    })
      .then(function (response) {
        dispatch(receiveService(serviceData.id, response.data.service, {}));
      })
      .catch(function (error) {
        dispatch(receiveService(serviceData.id, serviceData, {error: error}));
        throw error;
      });
  };
}

export function deleteService(serviceID) {
  return dispatch => {
    let variables = {
      id: serviceID,
    };

    return graphqlMutationPromise({
      mutation: deleteServiceQuery,
      variables: variables,
    }).then(response => {
      dispatch(receiveService(serviceID, undefined, {}));
    });
  };
}

export function createServiceAudit(service_id, audit_type, action, audit_value) {
  var timestamp = moment.utc().format();
  var variables = {
    service_audit: {
      service_id: service_id,
      audit_type: audit_type,
      action: action,
      audit_time: timestamp,
      audit_value: audit_value,
    },
  };
  return dispatch => {
    return graphqlMutationPromise({
      mutation: createServiceAuditQuery,
      variables: variables,
    }).then(response => {
      // Reload the service which just had an audit added
      dispatch(fetchService(service_id));
    });
  };
}
