// @format
import React from "react";
import PropTypes from "prop-types";
import { connect } from "react-redux";
import {
  Modal,
  ControlLabel,
  FormGroup,
  FormControl,
  Button,
  Alert,
} from "react-bootstrap";
import {
  setMessageBody,
  setMessageSubject,
  setMessageReplyTo,
  sendEmailAction,
} from "store/messaging";
import { toggleMessagingModal } from "store/uiReducer";
import { GetMessagingBaseURL } from "lib/env";
import WrapperTooltip from "components/WrapperTooltip";

function sendEmail(sendTo, replyTo, subject, body, service) {
  fetch(GetMessagingBaseURL() + "/email/", {
    method: "POST",
    body: JSON.stringify({
      Body: body,
      to: sendTo,
      Subject: subject,
      replyTo: replyTo.split(","),
      ServiceName: service.name,
      ServiceID: service.id,
    }),
    headers: {
      "Content-Type": "application/json",
    },
    credentials: "include",
  });
}

function sendSlack(sendTo, replyTo, subject, body, service) {
  fetch(GetMessagingBaseURL() + "/slack/", {
    method: "POST",
    body: JSON.stringify({
      Body: body,
      to: sendTo,
      Subject: subject,
      ServiceName: service.name,
      ServiceID: service.id,
    }),
    headers: {
      "Content-Type": "application/json",
    },
    credentials: "include",
  });
}

const messageTypeDict = {
  email: { title: "Email", shouldReplyTo: true, send: sendEmail },
  slack: { title: "Slack", shouldReplyTo: false, send: sendSlack },
};

// Validation functions

// validateReplyTo returns and object with a state property and
// a message property. The state will be one of 'success', 'warning', 'error',
// in alignment with bootstrap alert naming conventions
function validateReplyTo(email, shouldReplyTo) {
  // If we don't need a replyTo (i.e. Slack), then just say this is valid
  if (!shouldReplyTo) {
    return {
      state: "success",
      message: "",
    };
  }
  // Taken from https://stackoverflow.com/questions/46155/how-can-an-email-address-be-validated-in-javascript
  var re = /^(([^<>()\[\]\\.,;:\s@"]+(\.[^<>()\[\]\\.,;:\s@"]+)*)|(".+"))@((\[[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\])|(([a-zA-Z\-0-9]+\.)+[a-zA-Z]{2,}))$/;
  if (!re.test(String(email).toLowerCase())) {
    return {
      state: "error",
      message: "Must provide a valid Reply-To e-mail address",
    };
  }
  // If we get here, the e-mail must be valid. now check for twitch.tv or justin.tv replyto
  var domain = email.split("@")[1];
  if (
    domain !== "justin.tv" &&
    domain !== "twitch.tv" &&
    domain !== "amazon.com"
  ) {
    return {
      state: "error",
      message:
        "Reply-To e-mail address must be an internal account (@twitch.tv, @justin.tv, @amazon.com)",
    };
  }
  return {
    state: "success",
    message: "",
  };
}

// validateSubject ensures that the subject field
// is not empty. Returns an object with state and
// message properties. State will be one of 'success',
// 'warning', or 'error'
function validateSubject(subject) {
  if (subject === "") {
    return {
      state: "error",
      message: "Subject cannot be blank",
    };
  }
  return {
    state: "success",
    message: "",
  };
}

// validateBody is similar to validateSubject,
// but is meant to validate the message body.
// These two functions differ so that later, more
// complicated validations that differ between fields
// can be implemented.
function validateBody(body) {
  if (body === "") {
    return {
      state: "error",
      message: "Body cannot be blank",
    };
  }
  return {
    state: "success",
    message: "",
  };
}

// stateToStyle converts the name of the expected
// FormGroup validationState ('success', 'warning', 'error'),
// into a corresponding boostrap Alert bsStyle ('danger', 'warning', etc).
// In this case, we only care about 'danger' and 'warning' for
// this form
function stateToStyle(state) {
  if (state === "error") {
    return "danger";
  } else if (state === "warning") {
    return "warning";
  } else {
    return null;
  }
}

class MessagingModal extends React.Component {
  render() {
    const {
      subject,
      setSubject,
      replyTo,
      setReplyTo,
      body,
      setBody,
      sendTo,
      open,
      closeMessagingModal,
      messageType,
      titleFunc,
      service,
    } = this.props;
    let onHide = () => {
      closeMessagingModal();
      setReplyTo("");
      setSubject("");
      setBody("");
    };
    const { title, shouldReplyTo, send } = messageTypeDict[messageType];
    // Run validation
    var replyToErr = validateReplyTo(replyTo, shouldReplyTo);
    var subjectErr = validateSubject(subject);
    var bodyErr = validateBody(body);
    return (
      <Modal bsSize="lg" show={open} onHide={onHide}>
        <Modal.Header closeButton>
          <Modal.Title>{titleFunc(title)} </Modal.Title>
        </Modal.Header>
        <Modal.Body>
          <FormGroup controlId="emailSendTo">
            <ControlLabel>
              <WrapperTooltip title={"Send To"} />
            </ControlLabel>
            <FormControl.Static>{sendTo.toString()}</FormControl.Static>
          </FormGroup>
          {shouldReplyTo && (
            <FormGroup
              controlId="emailReplyTo"
              validationState={replyToErr.state}
            >
              <ControlLabel>
                <WrapperTooltip title={"Reply To"} />
              </ControlLabel>
              <FormControl
                type="email"
                value={replyTo}
                placeholder={"Enter " + title}
                onChange={e => {
                  setReplyTo(e.target.value);
                }}
              />
              <FormControl.Feedback />
              {replyToErr.state !== "success" && (
                <Alert bsStyle={stateToStyle(replyToErr.state)}>
                  {replyToErr.message}
                </Alert>
              )}
            </FormGroup>
          )}
          <FormGroup
            controlId="emailSubject"
            validationState={subjectErr.state}
          >
            <ControlLabel>
              <WrapperTooltip title={title + " Subject"} />
            </ControlLabel>
            <FormControl
              type="text"
              value={subject}
              placeholder={title + " Subject"}
              onChange={e => {
                setSubject(e.target.value);
              }}
            />
            <FormControl.Feedback />
            {subjectErr !== "success" && (
              <Alert bsStyle={stateToStyle(subjectErr.state)}>
                {subjectErr.message}
              </Alert>
            )}
          </FormGroup>
          <FormGroup controlId="emailBody" validationState={bodyErr.state}>
            <ControlLabel>
              <WrapperTooltip title={title + " Body"} />
            </ControlLabel>
            <FormControl
              componentClass="textarea"
              value={body}
              placeholder={title + " Body"}
              onChange={e => {
                setBody(e.target.value);
              }}
            />
            <FormControl.Feedback />
            {bodyErr !== "success" && (
              <Alert bsStyle={stateToStyle(bodyErr.state)}>
                {bodyErr.message}
              </Alert>
            )}
          </FormGroup>
        </Modal.Body>
        <Modal.Footer>
          <Button
            onClick={() => {
              send(sendTo, replyTo, subject, body, service);
              onHide();
            }}
            bsStyle="primary"
            disabled={
              replyToErr.state !== "success" ||
              subjectErr.state !== "success" ||
              bodyErr.state !== "success"
            }
          >
            Send {title}
          </Button>
          <Button onClick={onHide} bsStyle="danger">
            Cancel
          </Button>
        </Modal.Footer>
      </Modal>
    );
  }
}

var mapDispatchToProps = function(dispatch) {
  return {
    setSubject: subject => dispatch(setMessageSubject(subject)),
    setReplyTo: replyTo => dispatch(setMessageReplyTo(replyTo)),
    setBody: body => dispatch(setMessageBody(body)),
    closeMessagingModal: () => {
      dispatch(toggleMessagingModal(false));
    },
  };
};

var mapStateToProps = function(state, ownProps) {
  return {
    sendTo: state.messaging.sendTo,
    body: state.messaging.messageBody,
    replyTo: state.messaging.replyTo,
    subject: state.messaging.messageSubject,
    open: state.uiReducer.messagingModal.open,
    messageType: state.uiReducer.messagingModal.messageType,
    titleFunc: state.uiReducer.messagingModal.titleFunc,
    defaultReplyTo: state.uiReducer.messagingModal.defaultReplyTo,
    service: ownProps.service,
  };
};

export default connect(mapStateToProps, mapDispatchToProps)(MessagingModal);
