// @format
import React from "react";
import PropTypes from "prop-types";
import { connect } from "react-redux";
import { withRouter } from "react-router";
import { AutoSizer } from "react-virtualized";
import _ from "lodash";

import { fetchServiceIfNeeded } from "common/goracle/actions";
import DependencyTree from "common/DependencyTree/DependencyTree";

export class ServiceDependencyTree extends React.Component {
  createGraph() {
    const { service, servicesByID } = this.props;
    // Format is

    // var format = {
    //   "upstream" : {
    //     "children" : [
    //     "name": "origin",
    //     "children": [{
    //       "name": "upstream",
    //       "children": [{}]
    //     }]
    //   }
    // }
  }

  createUpstreamTree(
    serviceID,
    servicesByID,
    seen = {}
  ) {
    const serviceData = servicesByID ? servicesByID[serviceID] : undefined;
    const service = serviceData ? serviceData.service : undefined;
    // Create the upstream subtree rooted at self
    let tree = {
      service: service,
      url: `/services/${serviceID}`,
      repeated: false,
      children: [],
    };

    if (!service) {
      return tree;
    }

    if (seen[service.id]) {
      tree.repeated = true;
      return tree;
    }

    seen[service.id] = true;

    _.each(
      service.service_upstreams,
      function(upDep) {
        const depID = upDep.root_service.id;
        tree.children.push(
          this.createUpstreamTree(
            depID,
            servicesByID,
            seen
          )
        );
      }.bind(this)
    );
    return tree;
  }

  createDownstreamTree(
    serviceID,
    servicesByID,
    seen = {},
  ) {
    const serviceData = servicesByID ? servicesByID[serviceID] : undefined;
    const service = serviceData ? serviceData.service : undefined;
    // Create the downstream subtree rooted at self
    let tree = {
      service: service,
      repeated: false,
      children: [],
    };

    if (!service) {
      return tree;
    }

    if (seen[service.id]) {
      tree.repeated = true;
      return tree;
    }

    seen[service.id] = true;

    _.each(
      service.service_downstreams,
      function(downDep) {
        if (downDep.downstream_service) {
          const depID = downDep.downstream_service.id;
          tree.children.push(
            this.createDownstreamTree(
              depID,
              servicesByID,
              seen
            )
          );
        }
      }.bind(this)
    );
    return tree;
  }

  render() {
    const {
      serviceID,
      servicesByID,
      clickCallback,
    } = this.props;
    let upstreamData = this.createUpstreamTree(
      serviceID,
      servicesByID,
    );
    let downstreamData = this.createDownstreamTree(
      serviceID,
      servicesByID,
    );

    return (
      <AutoSizer disableHeight>
        {({ width }) => {
          if (width <= 0) {
            var w = Math.max(width, 800);
          } else {
            var w = width;
          }
          return (
            <DependencyTree
              upData={upstreamData}
              downData={downstreamData}
              clickCallback={clickCallback}
              height={600}
              width={w}
            />
          );
        }}
      </AutoSizer>
    );
  }
}

ServiceDependencyTree.propTypes = {
  serviceID: PropTypes.string,
  servicesByID: PropTypes.object.isRequired,
  clickCallback: PropTypes.func,
};

class ServiceDependencyTreeContainer extends React.Component {
  updateDependencyTree(servicesByID, serviceID, seen = {}) {
    const { fetchServiceIfNeeded } = this.props;
    const serviceData = servicesByID ? servicesByID[serviceID] : undefined;

    if (serviceID === undefined) {
      return;
    }

    // Don't do updates if we've already traversed this node
    if (seen[serviceID] == true) {
      return;
    }

    // Add to list of traversed nodes
    seen[serviceID] = true;

    // Starting from the service, iterate recursively through all upstreams and
    // downstreams, fetching any services that aren't in the graph yet.
    if (!serviceData) {
      fetchServiceIfNeeded(serviceID);
      return;
    }

    const service = serviceData.service;
    if (!service) {
      return;
    }

    _.each(
      service.service_upstreams,
      function(upDep) {
        if (upDep.root_service) {
          const depID = upDep.root_service.id;
          this.updateDependencyTree(servicesByID, depID, seen);
        }
      }.bind(this)
    );

    _.each(
      service.service_downstreams,
      function(downDep) {
        if (downDep.downstream_service) {
          const depID = downDep.downstream_service.id;
          this.updateDependencyTree(servicesByID, depID, seen);
        }
      }.bind(this)
    );
  }

  componentWillMount() {
    const { servicesByID, serviceID, fetchServiceIfNeeded } = this.props;

    this.updateDependencyTree(servicesByID, serviceID);
  }

  componentWillReceiveProps(nextProps) {
    const oldID = this.props.serviceID;
    const newID = nextProps.serviceID;
    const oldServicesByID = this.props.servicesByID;
    const newServicesByID = nextProps.servicesByID;

    if (oldID != newID || oldServicesByID != newServicesByID) {
      // Navigating to a new service or we've gotten new service data
      this.updateDependencyTree(newServicesByID, newID);
    }
  }
  componentDidUpdate() {
    const { servicesByID, serviceID } = this.props;

    this.updateDependencyTree(servicesByID, serviceID);
  }

  // As little presentational logic as possible goes here.
  // Use some basic conditionals to decide which visual components should be rendered
  // based on the current state
  render() {
    const {
      servicesByID,
      serviceID,
      history,
    } = this.props;
    let serviceData = servicesByID ? servicesByID[serviceID] : undefined;

    function defaultClickCallback(n) {
      let service = n.data.service;
      history.push(`/services/${service.id}`);
    }

    let clickCallback = this.props.clickCallback
      ? this.props.clickCallback
      : defaultClickCallback;

    return (
      <ServiceDependencyTree
        serviceID={serviceID}
        servicesByID={servicesByID}
        clickCallback={clickCallback}
      />
    );
  }
}

var mapDispatchToProps = function(dispatch) {
  return {
    fetchServiceIfNeeded: id => dispatch(fetchServiceIfNeeded(id, dispatch)),
  };
};

var mapStateToProps = function(state) {
  const { servicesByID } = state.goracle;

  return {
    servicesByID: servicesByID,
  };
};

ServiceDependencyTreeContainer.propTypes = {
  serviceID: PropTypes.string,
  clickCallback: PropTypes.func,
};

export default withRouter(
  connect(mapStateToProps, mapDispatchToProps)(ServiceDependencyTreeContainer)
);
