// __mocks__/graphql.js

import { ApolloClient } from "apollo-client";
import { SchemaLink } from "apollo-link-schema";
import { ApolloLink, from } from "apollo-link";
import { InMemoryCache } from "apollo-cache-inmemory";
import { makeExecutableSchema, addMockFunctionsToSchema } from "graphql-tools";
import _ from "lodash";

import { schemaString } from "./graphqlSchema";
import { mockFunctions, addMockGQLObject, removeMockGQLObject } from "./graphqlMockFunctions";

jest.unmock("graphql");

const mockGraphql = jest.genMockFromModule("lib/graphql");

let history = [];

function graphqlResetHistory() {
  history = [];
}

function graphqlGetHistory() {
  return history;
}

function graphqlPrintHistory(operationName) {
  _.each(history, h => {
    if (operationName && operationName != h.operationName) {
      return;
    }
    console.log(h.operationName, h.variables);
    console.log(JSON.stringify(h.result, undefined, 2));
  });
}

const schema = makeExecutableSchema({ typeDefs: schemaString });
const mocks = mockFunctions;
addMockFunctionsToSchema({ schema, mocks });

let instrumentationLink = new ApolloLink((operation, forward) => {
  return forward(operation).map((data) => {
    history.push({
      operationName: operation.operationName,
      variables: operation.variables,
      result: data
    });
    return data;
  })
}
);
let schemaLink = new SchemaLink({ schema });

// Turn off client caching for testing purposes
const defaultOptions = {
  watchQuery: {
    fetchPolicy: 'network-only',
    errorPolicy: 'all',
  },
  query: {
    fetchPolicy: 'network-only',
    errorPolicy: 'all',
  },
  mutate: {
    fetchPolicy: 'network-only',
    errorPolicy: 'all',
  },
}
const graphqlClient = new ApolloClient({
  link: from([instrumentationLink, schemaLink]),
  cache: new InMemoryCache(),
  defaultOptions: defaultOptions,
});

function graphqlQueryPromise({ query, variables }) {
  return graphqlClient
    .query({
      query: query,
      variables: variables,
      fetchPolicy: "network-only",
    })
}

function graphqlMutationPromise({ mutation, variables }) {
  return graphqlClient.mutate({
    mutation: mutation,
    variables: variables,
  });
}

function GraphqlQueryRaw(query, variables) {
  return graphqlClient.query({
    query: query,
    variables: variables,
    fetchPolicy: "network-only",
  });
}

mockGraphql.graphqlClient = graphqlClient;
mockGraphql.graphqlQueryPromise = graphqlQueryPromise;
mockGraphql.graphqlMutationPromise = graphqlMutationPromise;
mockGraphql.GraphqlQueryRaw = GraphqlQueryRaw;
mockGraphql.graphqlPrintHistory = graphqlPrintHistory;
mockGraphql.graphqlResetHistory = graphqlResetHistory;
mockGraphql.graphqlPrintHistory = graphqlPrintHistory;
mockGraphql.graphqlGetHistory = graphqlGetHistory;
mockGraphql.graphqlResetHistory = graphqlResetHistory;
mockGraphql.addMockGQLObject = addMockGQLObject;
mockGraphql.removeMockGQLObject = removeMockGQLObject;

module.exports = mockGraphql;
