import moment from "moment";
import _ from "lodash";
import { MockList } from "graphql-tools";

let idSequence = 1;
const slackDefaultID = "C8C8GEHEU";

// Return a monotonically increasing ID string
function nextID() {
  return `${idSequence++}`;
}

// For a type of object, if an ID is specified return either
// the hand-generated data or a default mock.
function instanceResolver(type, defaultID) {
  return (obj, args, context, info) => {
    let id = args.id || defaultID;
    let instance = _.get(data, `${type}.data[${id}]`, undefined);
    return instance || data[type].default(id);
  };
}

// For a type of object, return either the list of the hand-generated
// data or a list of 2 ascending mocks starting with ID 1
function listResolver(type) {
  return () => {
    try {
      let keys = data[type].data ? Object.keys(data[type].data) : [];
      let curKey = 0;
      return new MockList(keys.length || 2, (obj, args, context, info) => {
        let id = keys[curKey] || curKey + 1; // Start with 1 instead of 0
        curKey += 1;
        return instanceResolver(type, id)(obj, args, context, info);
      });
    } catch (err) {
      console.log(err);
    }
  }
}

export function addMockGQLObject(type, id, object) {
  data[type].data[id] = object;
}

export function removeMockGQLObject(type, id, object) {
  delete data[type].data[id];
}

let data = {
  Service: {
    default: id => {
      return {
        id: () => `${id}`,
        name: () => `Mock Service ${id}`,
        state: () => `Mock service state ${id}`,
        description: () => `Mock service description ${id}`,
        availability_objective: () => 5.0,
        team_id: "1",
        team: instanceResolver("Team", "1"),
        primary_owner_id: "3",
        primary_owner: instanceResolver("User", "3"),
        slack_channel_id: slackDefaultID,
        slack_channel: instanceResolver("SlackChannel", slackDefaultID),
        stores_user_data: true,
        access_requests: true,
        access_exception: "text",
        access_exception_validated_by: "test",
        deletion_requests: true,
        deletion_exception: "test",
        deletion_exception_validated_by: "test",
      };
    },
    data: {
      "1": {
        id: "1",
        name: "Root Service 1",
        state: "Active",
        description: "Root Service 1 Description",
        availability_objective: () => 5.0,
        slack_channel_id: slackDefaultID,
      },
      "2": {
        id: "2",
        state: "Active",
        name: "Downstream Service 2",
        description: "Downstream Service 2 Description",
        availability_objective: () => 5.0,
      },
    },
  },
  Team: {
    default: id => {
      return {
        id: () => `${id}`,
        name: () => `Mock Team ${id}`,
        description: () => `Mock team description ${id}`,
        services: listResolver("Service"),
        slack_channel_id: slackDefaultID,
        slack_channel: instanceResolver("SlackChannel", slackDefaultID)
      };
    },
  },
  User: {
    default: id => {
      return {
        employee_number: () => `${id}`,
        cn: () => `long User ${id}`,
        uid: () => `short${id}`,
        status: "active",
        inactive: false,
        preferred_name: () => 'Real Name ${id}',
        slack: () => {
          return {
            id: "slack424",
            name: "slackName"
          };
        },
        error: undefined,
      };
    },
  },
  Component: {
    default: id => {
      return {
        id: () => `${id}`,
        name: () => `Mock Component ${id}`,
        description: () => `Mock component description ${id}`,
        rollup: () => id % 2 == 0,
        service: instanceResolver("Service", "1"),
        metrics: listResolver("Metric"),
      };
    },
    data: {
      "1": {
        id: "1",
        name: "Fake component 1",
        service: instanceResolver("Service", "1"),
        metrics: listResolver("Metric"),
      },
      "2": {
        id: "2",
        name: "Fake component 2",
        service: instanceResolver("Service", "2"),
        metrics: listResolver("Metric"),
      },
    },
  },
  Account: {
    default: id => {
      return {
        id: () => `${id}`,
        name: () => `Mock Account ${id}`,
        aws_account_id: () => `mock_aws_account_id ${id}`,
        alias: () => `Mock Alias ${id}`,
        description: () => `Mock description ${id}`,
        org: () => instanceResolver("Org", "1"),
        org_id: "1",
        org: instanceResolver("Org", "1"),
        components: listResolver("Component"),
      };
    },
    data: {
      "1": {
        id: "1",
        name: "Fake account",
        alias: "Fake account alias",
        aws_account_id: "fake_aws_account_id_1",
        description: "fake description 1",
        org_id: "1",
        org: instanceResolver("Org", "1"),
      },
      "2": {
        id: "2",
        name: "Fake account 2",
        alias: "Fake account alias 2",
        aws_account_id: "fake_aws_account_id_2",
        description: "fake description 2",
        org_id: "2",
        org: instanceResolver("Org", "2"),
      },
    },
  },
  Org: {
    default: id => {
      return {
        id: () => `${id}`,
        name: () => `Mock Org ${id}`,
      };
    },
  },
  ServiceDependency: {
    default: id => {
      return {
        id: () => `${id}`,
        root_service_id: "1",
        downstream_service_id: "2",
      };
    },
  },
  SlackChannel: {
    default: id => {
      return {
        id: () => `${id}`,
        name: 'slackchannelName',
        topic: 'slackTopic',
        purpose: 'slackPurpose',
      };
    }
  },
  LogRecord: {
    default: id => {
      return {
        id: () => {
          return nextID();
        },
        created: () => moment().format(),
      };
    },
  },
  Metric: {
    default: id => {
      return {
        id: () => {
          return nextID();
        },
        calculation_type: "error_rate",
        latency_objective: 5.0,
        threshold: 50.0,
      };
    },
    data: {
      "1": {
        id: "1",
        name: "Fake error metric",
        calculation_type: "error_rate",
        latency_objective: 5.0,
        threshold: 50.0,
      },
      "2": {
        id: "2",
        name: "Fake latency metric",
        calculation_type: "latency",
        latency_objective: 5.0,
        threshold: 50.0,
      },
    },
  },
  Query: {
    default: id => {
      return {
        id: () => {
          return nextID();
        },
      };
    },
  },
};

export const mockFunctions = {
  ID: () => "1",
  Service: instanceResolver("Service", "111"),
  Team: instanceResolver("Team", "222"),
  Component: instanceResolver("Component", "333"),
  Account: instanceResolver("Account", "444"),
  Org: instanceResolver("Org", "555"),
  ServiceDependency: instanceResolver("ServiceDependency", "666"),
  User: instanceResolver("User", "777"),
  LogRecord: instanceResolver("LogRecord"),
  SlackChannel: instanceResolver("SlackChannel", slackDefaultID),
  QueryQL: () => ({
    accounts:  listResolver("Account"),
    services:  listResolver("Service"),
    servicesForOwner:  listResolver("Service"),
    teams:  listResolver("Team"),
    orgs:  listResolver("Org"),
    queries:  listResolver("Query"),
    metrics:  listResolver("Metric"),
    components:  listResolver("Component"),
    users:  listResolver("User"),
    slackChannels:  listResolver("SlackChannel"),
    // FIXME: Implement mock data for the rest of the list queries in queryql
  }),
};
