import Cookies from "js-cookie";

// local storage keys
export const sessionTokenKey = 'service-catalog:session';
export const sessionNextURL = 'service-catalog:nextURL';

export const errSessionInvalidated = new Error('session has been invalidated');
export const errUserNotAuthenticated = new Error('user is not authenticated');

export function getHashParams(url) {
  return (new URL(url))
    .hash
    .slice(1)
    .split('&')
    .map(x => x.split('='))
    .reduce((acc, [key, val]) => Object.assign(acc, {
      [decodeURIComponent(key)]: decodeURIComponent(val),
    }), {});
}

export default {
  // handle the oauth 2 completion callback.
  // - generates an access token
  // - stores the access token in local storage
  // - stores the access token in cookie
  handleCompletion: () => {
    const { access_token } = getHashParams(window.location);
    const encodedString = new Buffer(access_token).toString('base64');
    // https://stackoverflow.com/questions/18492576/share-cookie-between-subdomain-and-domain
    Cookies.set('scAuthToken', encodedString, { expires: 7, domain: 'internal.justin.tv' });
    window.localStorage.setItem(
      sessionTokenKey,
      JSON.stringify(access_token));
    let nextURL = window.sessionStorage.getItem(sessionNextURL)
    if (nextURL) {
      window.sessionStorage.removeItem(sessionNextURL);
      window.location = nextURL;
    } else {
      window.location.assign('/');
    }
  },

  // logoutUser will return a promise that will delete credentials
  logoutUser: () => {
    Cookies.remove('scAuthToken', { domain: 'internal.justin.tv' });
    Cookies.remove('scAuthUser', { domain: 'internal.justin.tv' });
    window.localStorage.removeItem(sessionTokenKey);
    window.sessionStorage.removeItem(sessionNextURL);
    window.location.assign('/');
  },

  // redirect user to the authentication page
  redirectToAuth: (config, options = {}) => {
    const {
      guardianClientID,
      guardianRedirectUrl,
      completionUrl,
    } = config;
    let { next } = options;

    if (!next) {
      next = window.location.href;
    }

    let url = `${guardianRedirectUrl}/authorize`;
    url += `?client_id=${guardianClientID}`;
    url += '&response_type=token';
    url += `&redirect_uri=${encodeURIComponent(completionUrl)}`;
    window.location.assign(url);
    window.sessionStorage.setItem(sessionNextURL, next);
  },
};
