// @format
import * as d3 from "d3";
import $ from "jquery";

// Layout for the dependency tree
//
// - svg
//    - outerContainer
//      - border (rect)
//      - treeContainer
//        - upTreeContainer
//        - downTreeContainer
//        - Labels
// - toolip

export default class D3DependencyTree {
  constructor({
    el,
    id,
    clickCallback,
    upData,
    downData,
    height = 400,
    width = 800,
  }) {
    this.upData = upData;
    this.downData = downData;
    this.clickCallback = clickCallback;
    this.el = el;
    this.id = id;
    this.svg = d3.select(this.el).select("#svg-" + this.id);
    this.totalHeight = height;
    this.totalWidth = width;
  }

  create(props) {
    let cd = this.chartDimensions();
    this.svg = d3
      .select(this.el)
      .append("svg")
      .attr("id", "svg-" + this.id)
      .attr("preserveAspectRatio", "none")
      .attr("width", cd.totalSize.width)
      .attr("height", cd.totalSize.height);

    // Create all of the container and graph elements
    let outer = this.svg.append("g").attr("class", "outerContainer");

    this.createTree();
    let tooltip = d3
      .select(this.el)
      .append("div")
      .attr("class", "tooltip")
      .style("opacity", 0);

    outer.on("mousemove", function() {
      // Stop propagation on all mouse events inside the SVG to the outside to
      // avoid generating an infinite loop.
      d3.event.stopPropagation();
    });

    this.update();
  }

  createTree() {
    let cd = this.chartDimensions();
    let outer = this.svg.select(".outerContainer");

    let treeContainer = outer.append("g").attr("class", "treeContainer");
    treeContainer.append("g").attr("class", "upstreamContainer");
    treeContainer.append("g").attr("class", "downstreamContainer");
    outer.append("rect").attr("class", "border");
    outer.append("g").attr("class", "labels");
    outer.append("g").attr("class", "legend");
  }

  update() {
    if (!this.upData && !this.downData) {
      return;
    }

    this.updateDimensions();
    let cd = this.chartDimensions();

    let treeContainer = this.svg.select(".treeContainer");

    let upstreamContainer = this.svg.select(".upstreamContainer");
    this.updateTree(
      upstreamContainer,
      cd.upTreeSize.width,
      cd.upTreeSize.height,
      this.upData,
      [[0, -1], [1, 0]]
    );

    let downstreamContainer = this.svg.select(".downstreamContainer");
    this.updateTree(
      downstreamContainer,
      cd.downTreeSize.width,
      cd.downTreeSize.height,
      this.downData,
      [[0, 1], [1, 0]]
    );

    let labels = this.svg.select(".labels");
    labels.selectAll("*").remove();
    labels
      .append("text")
      .attr("class", "dirlabel")
      .attr("dy", "1em")
      .attr("text-anchor", "start")
      .attr("transform", "translate(5, 5)")
      .text("Upstream Services");
    labels
      .append("text")
      .attr("class", "dirlabel")
      .attr("dy", "1em")
      .attr("text-anchor", "end")
      .attr("transform", "translate(" + (cd.totalSize.width - 5) + ",5)")
      .text("Downstream Services");

    this.updateLegend();

    let border = this.svg.select(".border");
    border
      .attr("width", cd.totalSize.width)
      .attr("height", cd.totalSize.height);
  }

  updateLegend() {
    let cd = this.chartDimensions();
    let legend = this.svg.select(".legend");
    legend.selectAll("*").remove();

    legend.attr(
      "transform",
      "translate(10," + (cd.totalSize.height - 160) + ")"
    );

    let scale = 40;

    // Border and background fill

    let fullLegend = legend
      .append("g")
      .attr("class", "fullLegend")
      .style("opacity", 0.0);
    fullLegend
      .append("rect")
      .attr("class", "graph-legend-box")
      .attr("width", 200)
      .attr("height", 150)
      .attr("pointer-events", "none");

    let contentCenter = fullLegend
      .append("g")
      .attr("transform", "translate(100, 75)");

    contentCenter
      .append("circle")
      .attr("class", "node")
      .attr("transform", "translate(-80,-40)")
      .attr("r", 0.25 * scale)
      .style("fill", "lightsteelblue");

    contentCenter
      .append("circle")
      .attr("class", "node")
      .attr("transform", "translate(-80,0)")
      .attr("r", 0.25 * scale)
      .style("fill", "#F00");


    contentCenter
      .append("circle")
      .attr("class", "node")
      .attr("transform", "translate(-80,+40)")
      .attr("r", 0.25 * scale)
      .style("fill", "#0F0");

    contentCenter
      .append("text")
      .attr("class", "graph-legend-text")
      .attr("text-anchor", "left")
      .attr("dy", ".35em")
      .attr("transform", "translate(-60,-40)")
      .text("Dep. Validation Pending");

    contentCenter
      .append("text")
      .attr("class", "graph-legend-text")
      .attr("text-anchor", "left")
      .attr("dy", ".35em")
      .attr("transform", "translate(-60,0)")
      .text("Dependencies Invalidated");

    contentCenter
      .append("text")
      .attr("class", "graph-legend-text")
      .attr("text-anchor", "left")
      .attr("dy", ".35em")
      .attr("transform", "translate(-60,40)")
      .text("Dependencies Validated");

    legend
      .append("rect")
      .attr("class", "graph-legend-box graph-legend-button")
      .attr("transform", "translate(0,130)")
      .attr("width", 60)
      .attr("height", 20)
      .on(
        "mouseover",
        function(d) {
          this.svg.select(".fullLegend").style("opacity", 1.0);
          this.svg.select(".graph-legend-button").style("opacity", 0.0);
        }.bind(this)
      )
      .on(
        "mouseout",
        function(d) {
          this.svg.select(".fullLegend").style("opacity", 0.0);
          this.svg.select(".graph-legend-button").style("opacity", 1.0);
        }.bind(this)
      );
    legend
      .append("text")
      .attr("text-anchor", "middle")
      .attr("transform", "translate(30,130)")
      .attr("dy", "1em")
      .attr("pointer-events", "none")
      .text("Legend");
  }

  updateDimensions() {
    let cd = this.chartDimensions();
    this.svg
      .attr("width", cd.totalSize.width)
      .attr("height", cd.totalSize.height);
    this.svg
      .select(".treeContainer")
      .attr(
        "transform",
        "translate(" + cd.treeMargins.x + "," + cd.treeMargins.y + ")"
      );
    this.svg
      .select(".upstreamContainer")
      .attr("transform", "translate(" + cd.treeSize.width / 2 + ",5)");
    this.svg
      .select(".downstreamContainer")
      .attr("transform", "translate(" + cd.treeSize.width / 2 + ",5)");
  }

  destroy() {
    // Clean up if necessary. No work here
  }

  chartDimensions() {
    let xMargin = 60;
    let yMargin = 15;
    let totalHeight = this.totalHeight;
    let totalWidth = this.totalWidth;
    let treeWidth = totalWidth - 2 * xMargin;
    let treeHeight = totalHeight - 2 * yMargin;
    return {
      totalSize: {
        width: totalWidth,
        height: totalHeight,
      },
      treeMargins: {
        x: xMargin,
        y: yMargin,
      },
      treeSize: {
        width: treeWidth,
        height: treeHeight,
      },
      upTreeSize: {
        width: treeWidth / 2,
        height: treeHeight,
      },
      downTreeSize: {
        width: treeWidth / 2,
        height: treeHeight,
      },
    };
  }

  updateTree(treeContainer, width, height, data, transform) {
    // Note: X and Y are transposed on this graph

    let cd = this.chartDimensions();
    let tree = d3.tree().size([height, width]);
    let root = d3.hierarchy(data, function(d) {
      return d.children;
    });

    let treeData = tree(root);
    let nodes = treeData.descendants();
    let links = treeData.descendants().slice(1);

    treeContainer.selectAll(".node").remove();
    treeContainer.selectAll(".link").remove();
    treeContainer.selectAll(".label").remove();

    function tX(x, y) {
      return transform[0][0] * x + transform[0][1] * y;
    }

    function tY(x, y) {
      return transform[1][0] * x + transform[1][1] * y;
    }

    root.x = height / 2;

    // Update
    let i = 0;
    // Update nodees
    let node = treeContainer.selectAll(".node").data(nodes, function(d) {
      return d.id || (d.id = ++i);
    });

    // Update the links...
    let link = treeContainer.selectAll("path.link").data(links, function(d) {
      return d.id;
    });

    // Enter any new links at the parent's previous position.
    let linkEnter = link
      .enter()
      .append("path", "g")
      .attr("class", "link")
      .attr("d", function(d) {
        return diagonal(d, d.parent);
      });

    // Creates a curved (diagonal) path from parent to the child nodes
    function diagonal(s, d) {
      let path = `M ${tX(s.x, s.y)} ${tY(s.x, s.y)}
            C ${(tX(s.x, s.y) + tX(d.x, d.y)) / 2} ${tY(s.x, s.y)},
              ${(tX(s.x, s.y) + tX(d.x, d.y)) / 2} ${tY(d.x, d.y)},
              ${tX(d.x, d.y)} ${tY(d.x, d.y)}`;

      return path;
    }

    let el = this.el;
    let svg = this.svg;

    let nodePos = node
      .enter()
      .append("g")
      .attr("class", "node")
      .attr("transform", function(d) {
        return "translate(" + tX(d.x, d.y) + "," + tY(d.x, d.y) + ")";
      })
      .on("click", this.clickCallback);

    let nodeSize = Math.max(
      Math.min(10, cd.treeSize.height / (3 * nodes.length)),
      1
    );

    let drawLabel = nodeSize > 7;

    function validated(d, auditType) {
      let latest_audits = d.data.service
        ? d.data.service.latest_service_audits
        : undefined;
      if (latest_audits === undefined) {
        return undefined;
      }

      for (let i = 0; i < latest_audits.length; ++i) {
        let a = latest_audits[i];
        if (a.audit_type === auditType) {
          return a.action === "validated";
        }
      }
      return undefined;
    }

    // Add circles for validation
    nodePos
      .append("circle") // Add Circle for the nodes
      .attr("transform", function(d) {
        return "translate(" + nodeSize + "," + nodeSize + ")";
      })
      .attr("r", function(d) {
        if (d === root) {
          return 5;
        }
        return nodeSize / 2;
      })
      .style("fill", function(d) {
        switch (validated(d, "dependencies")) {
          case true:
            return "#0f0";
          case false:
            return "#f00";
          case undefined:
            return "lightsteelblue";
        }
      })
      .on("mouseover", function(d) {
        d3.select(this).attr("class", "hover");
      })
      .on("mouseout", function(d) {
        d3.select(this).attr("class", undefined);
      });

    nodePos
      .append("circle")
      .attr("r", function(d) {
        if (d === root) {
          return 10;
        }
        return nodeSize;
      })
      .style("fill", "#CCC")
      .on("mouseover", function(d) {
        d3.select(this).attr("class", "hover");
        let tooltip = d3.select(el).select(".tooltip");

        if (drawLabel) {
          // Don't display tooltip if node is labeled
          return;
        }

        tooltip.html(d.data.service ? d.data.service.name : "unknown");

        let modalParent = $(svg.node()).parents(".modal-dialog")[0];
        let leftOffset = modalParent
          ? -modalParent.getBoundingClientRect().left
          : 0;
        let topOffset = modalParent
          ? -modalParent.getBoundingClientRect().top
          : 0;

        let svgBounds = svg.node().getBoundingClientRect();

        leftOffset += svgBounds.x + cd.treeMargins.x + cd.treeSize.width / 2;
        leftOffset += -0.5 * tooltip.node().getBoundingClientRect().width;
        topOffset += svgBounds.y + cd.treeMargins.y + nodeSize + 10;

        tooltip
          .style("left", `${leftOffset + tX(d.x, d.y)}px`)
          .style("top", `${topOffset + tY(d.x, d.y)}px`)
          .style("opacity", 1.0);
      })
      .on("mouseout", function(d) {
        d3.select(this).attr("class", undefined);
        if (drawLabel) {
          return;
        }
        let tooltip = d3.select(el).select(".tooltip");
        tooltip.style("opacity", 0.0);
      });

    if (drawLabel) {
      nodePos
        .append("text") // Add labels for the nodes
        .attr("dy", ".35em")
        .attr("y", nodeSize * 2 + 3)
        .attr("text-anchor", "middle")
        .text(function(d) {
          if (d === root) {
            // Don't render name of root node
            return "";
          }
          return d.data.service ? d.data.service.name : "--";
        });
    }
  }
}
