import React from "react";
import {Link} from "react-router-dom";
import _ from "lodash";
import moment from "moment";
import Icon from "react-icons-kit";
import {slack} from 'react-icons-kit/fa/slack'
import {getUserDisplayName} from 'common/User/Util';
import {getPrimaryOwnerData} from "../common/ServiceTable/components/ServiceTable";
import {profile} from "react-icons-kit/icomoon/profile";
import ManagerInfoPopover from "../common/User/UserView/ManagerInfoPopover"

const base = "/";

export function toService(id) {
  return `${base}services/${id}`;
}

export function toOrg(id) {
  return `${base}orgs/${id}`;
}

export function toBase(path) {
  return `${base}${path}`;
}

export function toEmail(path) {
  return "mailto:" + path;
}

export function toEmployeeEmail(employeeEID) {
  return `mailto:${employeeEID}@twitch.tv`;
}

export function toSlack(path) {
  return "https://twitch.slack.com/app_redirect?channel=" + path.replace(/^#/, "");
}

export function toPagerDuty(path) {
  return path;
}

export function generateValidationFormatter(validationField, showAuditor = false, showAuditValue = false, showTime = false, extraContentGenerator = null) {
  return function (s) {
    if (!s.latest_service_audits) {
      return {
        display: "--",
        value: "",
      };
    }
    let audits = s.latest_service_audits;
    let extraContent = null;
    if (extraContentGenerator != null) {
      extraContent = extraContentGenerator(s);
    }
    for (let i = 0; i < audits.length; i++) {
      let audit = audits[i];
      if (audit.audit_type === validationField) {
        let momentTime = "";
        if (showTime) {
          momentTime = " " + moment(audit.audit_time).fromNow();
        }

        let auditor = "";
        if (showAuditor) {
          auditor = ` by ${audit.auditor}`;
        }

        let auditValue = "";
        if (showAuditValue) {
          auditValue = ` ${audit.audit_value}`;
        }

        let style, validateText = "";
        if (audit.action === "validated") {
          style = "success";
          validateText = "Validated";
        } else if (audit.action === "invalidated") {
          style = "danger";
          validateText = "Invalidated";
        }
        return {
          value: validateText,
          display: <div className={"text-" + style}>
            {validateText + auditValue + auditor + momentTime}
            {extraContent}
          </div>,
        };
      }
    }
    return {
      value: "Pending",
      display: <div className={"text-muted"}>Pending{extraContent}</div>,
    };
  }
}

var primaryOwnerValidationFormatter = generateValidationFormatter('owner');

export {primaryOwnerValidationFormatter};

export const serviceFormatter = s => {
  if (!s) {
    return {
      display: "--",
      value: "",
    };
  }
  var link = toService(s.id);
  return {
    display: <Link to={link}>{s.name}</Link>,
    value: s.name,
  };
};

export function tierFormatter(s) {
  if (s.attributes) {
    for (let attr of s.attributes) {
      if (attr.name === "tier") {
        return {
          display: attr.value,
          value: attr.value,
        };
      }
    }
  }
  return {
    display: " ",
    value: "Unspecified",
  };
}

export function stateFormatter(s) {
  let state = _.get(s, "state", null);
  if (state !== null) {
    if (state === "Active") {
      return {
        display: "Active",
        value: "Active",
      }
    } else {
      return {
        display: "Inactive",
        value: "Inactive",
      }
    }

  } else {
    return {
      display: "Unknown",
      value: "Unknown",
    };
  }
}

export function primaryOwnerFormatter() {
  return function (s) {
    let po = _.get(s, "primary_owner", null);
    if (po !== null) {
      return {
        display: getUserDisplayName(po),
        value: getUserDisplayName(po),
      }
    } else {
      return {
        display: "Unknown",
        value: "Unknown",
      };
    }
  }
}

export function slackFormatter(s) {
  let po = _.get(s, "primary_owner", null);
  if (po !== null) {
    if (po.slack) {
      var poSlack = <a target="_blank"
                       href={"https://twitch.slack.com/app_redirect?channel=" + po.slack.id}>{po.slack.name}</a>;
    }
    return {
      display: <div><Icon size={12} icon={slack}/><span> {poSlack}</span></div>,
      value: poSlack,
    }
  } else {
    return {
      display: "Unknown",
      value: "Unknown",
    };
  }
}

export function managerFormatter(s) {
  let po = _.get(s, "primary_owner", null);
  if (po !== null) {
    let primaryOwnerData = getPrimaryOwnerData(po.uid);
    if (primaryOwnerData) {
      var managerValue = (<div>
          {primaryOwnerData.manager_preferred_name}
          <a target="_blank">
            {<Icon className={'pull-right'} icon={profile}/>}
          </a>
        </div>
      );
      var display = (<div>
          <ManagerInfoPopover id={po.uid} user_id={po.uid} trigger={managerValue}/>
        </div>
      );
    } else {
      display = ""
    }
    return {
      display: display,
      value: primaryOwnerData ? primaryOwnerData.manager_preferred_name : "",
    }
  } else {
    return {
      display: "Unknown",
      value: "Unknown",
    };
  }
}

export function teamFormatter(s) {
  let po = _.get(s, "primary_owner", null);
  if (po !== null) {
    let primaryOwnerData = getPrimaryOwnerData(po.uid)
    return {
      display: primaryOwnerData ? primaryOwnerData.team_name : "",
      value: primaryOwnerData ? primaryOwnerData.team_name : "",
    }
  } else {
    return {
      display: "Unknown",
      value: "Unknown",
    };
  }
}

export function orgFormatter(s) {
  let po = _.get(s, "primary_owner", null);
  if (po !== null) {
    let primaryOwnerData = getPrimaryOwnerData(po.uid)
    return {
      display: primaryOwnerData ? primaryOwnerData.org_name : "",
      value: primaryOwnerData ? primaryOwnerData.org_name : "",
    }
  } else {
    return {
      display: "Unknown",
      value: "Unknown",
    };
  }
}

export function buFormatter(s) {
  let po = _.get(s, "primary_owner", null);
  if (po !== null) {
    let primaryOwnerData = getPrimaryOwnerData(po.uid)
    return {
      display: primaryOwnerData ? primaryOwnerData.bu_name : "",
      value: primaryOwnerData ? primaryOwnerData.bu_name : "",
    }
  } else {
    return {
      display: "Unknown",
      value: "Unknown",
    };
  }
}

export var serviceColumns = [
  {
    title: "Name",
    key: "name",
    filter: "text",
    isBasicFilter: true,
    formatter: function (service) {
      if (!service) {
        return {
          display: "Missing service",
          value: "Missing service",
        };
      }
      return {
        display: <Link to={toService(service.id)}>{service.name}</Link>,
        value: service.name,
      };
    },
  },
  {
    title: "Primary Owner",
    key: "primary_owner.cn",
    formatter: primaryOwnerFormatter(false),
    filter: "select",
    isBasicFilter: true,
  },
  {
    title: "Service State",
    key: "state",
    formatter: stateFormatter,
    filter: "select",
    isBasicFilter: true,
  },
  {
    title: "Tier",
    key: "tier", // doesnt matter
    formatter: tierFormatter,
  },
  {
    title: "Primary Owner Slack",
    key: "primary_owner.slack",
    formatter: slackFormatter,
  },
  {
    title: "Manager",
    key: "primary_owner.manager",
    formatter: managerFormatter,
    filter: "select",
    isBasicFilter: false,
  },
  {
    title: "Team",
    filter: "select",
    isBasicFilter: false,
    key: "team_name",
    formatter: teamFormatter,
  },
  {
    title: "Org",
    filter: "select",
    isBasicFilter: false,
    key: "org_name",
    formatter: orgFormatter,
  },
  {
    title: "Business Unit",
    filter: "select",
    isBasicFilter: false,
    key: "bu_name",
    formatter: buFormatter,
  },
  {
    title: "Primary Owner Validation",
    key: "latest_service_audits",
    filter: "select",
    isBasicFilter: true,
    formatter: primaryOwnerValidationFormatter,
  },
  {
    title: "Attributes",
    filter: "attributes",
    isBasicFilter: true,
    key: "attributes",
  },
];

export var dependencyServiceColumns = [
  {
    title: "Name",
    key: "name",
    filter: "text",
    isBasicFilter: true,
    formatter: function (service) {
      if (!service) {
        return {
          display: "Missing service",
          value: "Missing service",
        };
      }
      return {
        display: <Link to={toService(service.id)}>{service.name}</Link>,
        value: service.name,
      };
    },
  },
  {
    title: "Primary Owner",
    key: "primary_owner.cn",
    formatter: primaryOwnerFormatter(false),
    filter: "select",
    isBasicFilter: true,
  },
  {
    title: "Service State",
    key: "state",
    formatter: stateFormatter,
    filter: "select",
    isBasicFilter: true,
  },
  {
    title: "Tier",
    key: "tier", // doesnt matter
    formatter: tierFormatter,
  },
  {
    title: "Primary Owner Slack",
    key: "primary_owner.slack",
    formatter: slackFormatter,
  },
  {
    title: "Primary Owner Validation",
    key: "latest_service_audits",
    filter: "select",
    isBasicFilter: false,
    formatter: primaryOwnerValidationFormatter,
  },
  {
    title: "Attributes",
    filter: "attributes",
    isBasicFilter: false,
    key: "attributes",
  },
];
