import { GetAPIBaseURL } from "./env";
import { ApolloClient } from "apollo-client";
import { createHttpLink } from "apollo-link-http";
import { InMemoryCache } from "apollo-cache-inmemory";
import { reportError } from "common/ErrorHandling/Actions";
import { store } from "../main";

import gql from "graphql-tag";

export const graphqlClient = new ApolloClient({
  link: createHttpLink({
    uri: GetAPIBaseURL(),
    credentials: "include",
  }),
  cache: new InMemoryCache(),
});

export function graphqlQueryPromise({ query, variables }) {
  return graphqlClient
    .query({ query: query, variables: variables, fetchPolicy: "network-only" })
    .catch(error => {
      console.log(error);
      console.trace();
      throw error;
    });
}

export function graphqlMutationPromise({ mutation, variables }) {
  return graphqlClient
    .mutate({ mutation: mutation, variables: variables })
    .catch(error => {
      console.log(error);
      console.trace();
      store.store.dispatch(reportError(error.message));
      throw error;
    });
}

// Takes a query, a dispatcher, and some action firing
// methods and will do a GraphQL query, firing the right
// actions as needed
export function GraphqlQuery(
  graphqlQuery,
  graphqlVariables,
  loadingAction,
  errorAction,
  successAction,
  dispatch
) {
  dispatch(loadingAction(true));
  return graphqlClient
    .query({ query: graphqlQuery, variables: graphqlVariables })
    .then(response => dispatch(successAction(response.data)))
    .then(response => {
      dispatch(loadingAction(false));
      return response;
    })
    .catch(error => {
      console.log(error.message);
      dispatch(loadingAction(false));
      dispatch(errorAction(true));
    });
}

// Takes a mutation, and executes it, working through the returned
// promise and eventually handing back the results
// Takes loading and error actions and fires them as appropriate
export function GraphqlMutation(
  graphqlMutation,
  graphqlVariables,
  loadingAction,
  errorAction,
  successAction,
  dispatch,
  refetchQueries = []
) {
  dispatch(loadingAction(true));
  return graphqlClient
    .mutate({
      mutation: graphqlMutation,
      variables: graphqlVariables,
      refetchQueries,
    })
    .then(response => dispatch(successAction(response.data)))
    .then(response => {
      dispatch(loadingAction(false));
      return response;
    })
    .catch(error => {
      console.log(error.message);
      dispatch(loadingAction(false));
      dispatch(errorAction(true));
      throw error;
    });
}

function combineMutations(mutationName, mutationData) {
  var vars = {};
  var varsString = "";
  var mutationString = "";
  mutationData.forEach(mutation => {
    varsString += mutation.varsString;
    mutationString += mutation.mutationString;
    Object.assign(vars, mutation.vars);
  });
  // return { varsString: varsString, mutationString: mutationString, vars: vars };
  return {
    mutation: gql`mutation ${mutationName}(${varsString}) {${mutationString}}`,
    variables: vars,
  };
}
// takes an array of mutations of the form { varsString: varsString, mutationString: mutationString, vars: vars }
// combines them and executes them as one query
export function GraphqlMutationCombined(
  mutationName,
  mutations,
  loadingAction,
  errorAction,
  successAction,
  dispatch
) {
  dispatch(loadingAction(true));
  graphqlClient
    .mutate(combineMutations(mutationName, mutations))
    .then(response => dispatch(successAction(response.data)))
    .then(response => {
      dispatch(loadingAction(false));
      return response;
    })
    .catch(error => {
      console.log(error.message);
      dispatch(loadingAction(false));
      dispatch(errorAction(true));
    });
}

// A GraphQL query implementation that does no action dispatching on behalf of the caller;
// instead, it just returns the raw results
export function GraphqlQueryRaw(graphqlQuery, graphqlVariables) {
  return graphqlClient
    .query({ query: graphqlQuery, variables: graphqlVariables })
    .then(response => response.data)
    .catch(error => {
      console.log(error.message);
      return [{ label: "Error fetching options", value: null }];
    });
}
