import React from "react";
import gql from "graphql-tag";
import PropTypes from "prop-types";
import { Query } from "react-apollo";
import Spinner from "react-spinkit";
import ErrorDisplay from "../../components/ErrorDisplay";
import LogReportsTable from "./LogReportsView/LogReportsTable";
import { Col, FormControl, FormGroup, HelpBlock, ToggleButton, ToggleButtonGroup } from "react-bootstrap";
import { WrapperTooltip } from "../../components/WrapperTooltip";
import LoginComponent from "../Login/LoginComponent";

const logsQuery = gql `
  query logs($limit: Int!, $offset: Int!) {
    logs(limit: $limit, offset: $offset) {
      id
      created
      author
      action
      item_type
      item_label
      item_id
      before
      after
    }
  }
`

var filterAutoEnabled = false;

export function isFilterAutoEnabled() {
  return filterAutoEnabled;
}

class LogReportsPage extends React.Component {

  constructor(props, context) {
    super(props, context);
    this.handleFilterAutoChange = this.handleFilterAutoChange.bind(this);
    this.handleLimitChange = this.handleLimitChange.bind(this);
    this.handleOffsetChange = this.handleOffsetChange.bind(this);
    this.state = {
      isFilterAutoEnabled: 0,
      limit: props.limit,
      offset: props.offset,
    };
  }

  handleFilterAutoChange(e) {
    if (e === 1) {
      filterAutoEnabled = true
    } else {
      filterAutoEnabled = false
    }
    this.setState({ isFilterAutoEnabled: e });
  }

  handleLimitChange(e) {
    let parsedLimit = parseInt(e.target.value, 10)
    if (parsedLimit > 0) {
      this.setState({ limit: parsedLimit });
    } else {
      this.setState({ limit: 100 });
    }
  }

  handleOffsetChange(e) {
    let parsedOffset = parseInt(e.target.value, 10)
    if (parsedOffset > -1) {
      this.setState({ offset: parsedOffset });
    } else {
      this.setState({ offset: 0 });
    }
  }

  render() {
    let {limit, offset} = this.state;
    return (
      <div>
        <LoginComponent/>
        <form>
          <FormGroup
            controlId="formLimit"
          >
            <Col md={4}>
              <HelpBlock>Max number of entries:</HelpBlock>
              <FormControl
                type="number"
                limit={this.state.limit}
                placeholder="100"
                onChange={this.handleLimitChange}
              />
              <FormControl.Feedback />
            </Col>
            <Col md={4}>
              <HelpBlock>Starting offset:</HelpBlock>
              <FormControl
                type="number"
                limit={this.state.offset}
                placeholder="0"
                onChange={this.handleOffsetChange}
              />
              <FormControl.Feedback />
            </Col>
            <Col md={4}>
              <HelpBlock>View logs created by Automation:</HelpBlock>
              <ToggleButtonGroup
                type="radio"
                name="filter-auto-options"
                defaultValue={0}
                onChange={this.handleFilterAutoChange}
              >
                <ToggleButton style={{color: 'purple'}} value={0}><WrapperTooltip title={"Show"}/></ToggleButton>
                <ToggleButton style={{color: 'purple'}} value={1}><WrapperTooltip title={"Hide"}/></ToggleButton>
              </ToggleButtonGroup>
            </Col>
          </FormGroup>
        </form>
        <Query query={logsQuery} variables={{ "limit": limit, "offset": offset }}>
          {({ loading, error, data}) => {
            if (loading) {
              return <Spinner fadeIn={"quarter"} name={"pacman"} color={"purple"}/>
            }
            if (error) {
              console.log(error);
              console.trace();
              return <ErrorDisplay errorMessage={error.message}/>
            }
            let logs = data.logs;
            if (!logs || logs === undefined || logs.length === 0) {
              return <div>Could not find any logs...</div>
            }
            return <LogReportsTable isFilterAutoEnabled={this.state.isFilterAutoEnabled} logs={logs} limit={limit} offset={offset}/>;
          }}
        </Query>
      </div>
    )
  }
}

LogReportsPage.propTypes = {
  limit: PropTypes.number,
  offset: PropTypes.number,
  loading: PropTypes.bool,
  error: PropTypes.object,
}

LogReportsPage.defaultProps = {
  limit: 100,
  offset: 0,
}

export default LogReportsPage;
